pub mod test_helpers;

use crate::test_helpers::*;
use glql::utils::feature_flags::FeatureFlag;

#[test]
fn test_milestone_equals_value() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and milestone = \"17.5\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(milestoneTitle: \"17.5\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and milestone = \"17.5\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(milestoneTitle: \"17.5\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_milestone_equals_list() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and milestone = (\"Backlog\", \"17.5\")"),
        "Error: `milestone` does not support the equals (`=`) operator for `(\"Backlog\", \"17.5\")`. Supported operators: is one of (`in`), not equals (`!=`)."
    );
}

#[test]
fn test_milestone_in_list() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and milestone in (\"Backlog\", \"17.5\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(milestoneTitle: [\"Backlog\", \"17.5\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and milestone in (\"Backlog\", \"17.5\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(milestoneTitle: [\"Backlog\", \"17.5\"], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_milestone_not_equals_list() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and milestone != (\"Backlog\", \"17.5\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(not: {milestoneTitle: [\"Backlog\", \"17.5\"]}, before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and milestone != (\"Backlog\", \"17.5\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(not: {milestoneTitle: [\"Backlog\", \"17.5\"]}, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_milestone_with_reference_syntax() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and milestone = %Backlog")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(milestoneTitle: \"Backlog\", before: $before, after: $after, first: $limit) {",
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and milestone = %Backlog")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(milestoneTitle: \"Backlog\", before: $before, after: $after, first: $limit) {",
    );
}

#[test]
fn test_milestone_with_reference_syntax_in_array() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and milestone in (%Backlog, %\"Awaiting further demand\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(milestoneTitle: [\"Backlog\", \"Awaiting further demand\"], before: $before, after: $after, first: $limit) {",
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and milestone in (%Backlog, %\"Awaiting further demand\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(milestoneTitle: [\"Backlog\", \"Awaiting further demand\"], before: $before, after: $after, first: $limit) {",
    );
}

#[test]
fn test_milestone_equals_username() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and milestone = @username"),
        "Error: `milestone` cannot be compared with `@username`. Supported value types: `String`, `List`, `Nullable` (`null`, `none`, `any`), `Enum` (`upcoming`, `started`), `Milestone` (example: `%Backlog`)."
    );
}

#[test]
fn test_invalid_milestone_reference() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and milestone = %milestone#1"),
        "Error: Invalid milestone reference `%milestone#1`"
    );
}
