pub mod test_helpers;

use crate::test_helpers::*;
use glql::utils::feature_flags::FeatureFlag;

#[test]
fn test_label_equals_string() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = \"foo\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(labelName: \"foo\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = \"foo\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(labelName: \"foo\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_label_equals_array() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = (\"foo\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(labelName: [\"foo\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = (\"foo\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(labelName: [\"foo\"], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_label_equals_empty_array() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = ()")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(labelName: \"NONE\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = ()")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(labelName: \"NONE\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_label_not_equals_empty_array() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label != ()")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(labelName: \"ANY\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label != ()")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(labelName: \"ANY\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_label_multiple_complex_example() {
    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and label != \"foo\" and label != \"bar\" and label = \"baz\" and label = \"quo\" and label = (\"bug\", \"enhancement\") and label != (\"feature\", \"maintenance\")"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    issues(labelName: [\"baz\", \"bug\", \"enhancement\", \"quo\"], not: {labelName: [\"bar\", \"feature\", \"foo\", \"maintenance\"]}, before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and label != \"foo\" and label != \"bar\" and label = \"baz\" and label = \"quo\" and label = (\"bug\", \"enhancement\") and label != (\"feature\", \"maintenance\")"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    workItems(labelName: [\"baz\", \"bug\", \"enhancement\", \"quo\"], not: {labelName: [\"bar\", \"feature\", \"foo\", \"maintenance\"]}, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_label_equals_null() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = null")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(labelName: \"NONE\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = null")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(labelName: \"NONE\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_label_in() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label in (\"foo\", \"bar\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(or: {labelNames: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label in (\"foo\", \"bar\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(or: {labelNames: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_label_multiple_operator() {
    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and label in (\"foo\", \"bar\") and label = \"baz\""
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    issues(labelName: \"baz\", or: {labelNames: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and label in (\"foo\", \"bar\") and label = \"baz\""
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    workItems(labelName: \"baz\", or: {labelNames: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_label_two_equals_string() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = \"foo\" and label = \"bar\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(labelName: [\"foo\", \"bar\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = \"foo\" and label = \"bar\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(labelName: [\"foo\", \"bar\"], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_label_multiple_equals_string() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = \"foo\" and label = \"bar\" and label=\"boo\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(labelName: [\"foo\", \"bar\", \"boo\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = \"foo\" and label = \"bar\" and label=\"boo\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(labelName: [\"foo\", \"bar\", \"boo\"], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_label_equals_string_array() {
    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and label = \"foo\" and label = (\"bar\", \"boo\")"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    issues(labelName: [\"bar\", \"boo\", \"foo\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and label = \"foo\" and label = (\"bar\", \"boo\")"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    workItems(labelName: [\"bar\", \"boo\", \"foo\"], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_label_multiple_equals_string_array() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = \"foo\" and label = (\"bar\", \"boo\") and label=\"baz\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(labelName: [\"bar\", \"boo\", \"foo\", \"baz\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = \"foo\" and label = (\"bar\", \"boo\") and label=\"baz\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(labelName: [\"bar\", \"boo\", \"foo\", \"baz\"], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_label_multiple_equals_array() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = (\"foo\", \"bar\") and label=(\"boo\", \"baz\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(labelName: [\"bar\", \"baz\", \"boo\", \"foo\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = (\"foo\", \"bar\") and label=(\"boo\", \"baz\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(labelName: [\"bar\", \"baz\", \"boo\", \"foo\"], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_label_not_equals() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label != \"foo\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(not: {labelName: \"foo\"}, before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label != \"foo\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(not: {labelName: \"foo\"}, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_label_with_reference_equals_string() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = ~\"foo\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(labelName: \"foo\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = ~\"foo\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(labelName: \"foo\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_label_with_reference_multiple_equals_array() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = (~frontend, ~AI) and label=(~\"1st contribution\", ~\"workflow::in review\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(labelName: [\"1st contribution\", \"AI\", \"frontend\", \"workflow::in review\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = (~frontend, ~AI) and label=(~\"1st contribution\", ~\"workflow::in review\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(labelName: [\"1st contribution\", \"AI\", \"frontend\", \"workflow::in review\"], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_invalid_label_equals_username() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = @username"),
        "Error: `label` cannot be compared with `@username`. Supported value types: `String`, `List`, `Nullable` (`null`, `none`, `any`), `Label` (example: `~frontend`)."
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = @username"),
        "Error: `label` cannot be compared with `@username`. Supported value types: `String`, `List`, `Nullable` (`null`, `none`, `any`), `Label` (example: `~frontend`)."
    );
}

#[test]
fn test_invalid_label_reference() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = ~\"unterminated string"),
        "Error: Invalid label reference `~\"unterminated string`"
    );
}

#[test]
fn test_label_equals_mixed_references() {
    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and label = ~foo and label = (~bar, ~baz)"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    issues(labelName: [\"bar\", \"baz\", \"foo\"], before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = ~foo and label = ~bar")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(labelName: [\"foo\", \"bar\"], before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and label = \"foo\" and label = (~bar, ~baz)"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    issues(labelName: [\"bar\", \"baz\", \"foo\"], before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = \"foo\" and label = (\"bar\", ~baz) and label = ~quox")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(labelName: [\"bar\", \"baz\", \"foo\", \"quox\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = \"foo\" and label = (\"bar\", ~baz) and label = ~quox")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(labelName: [\"bar\", \"baz\", \"foo\", \"quox\"], before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = ~foo and label = ~bar")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(labelName: [\"foo\", \"bar\"], before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and label = \"foo\" and label = (~bar, ~baz)"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    workItems(labelName: [\"bar\", \"baz\", \"foo\"], before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and label = \"foo\" and label = (\"bar\", ~baz) and label = ~quox")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(labelName: [\"bar\", \"baz\", \"foo\", \"quox\"], before: $before, after: $after, first: $limit) {"
    );
}
