#[cfg(test)]
pub mod test_helpers;

use crate::test_helpers::*;
use glql::utils::feature_flags::FeatureFlag;

#[test]
fn test_iteration_equals_value() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and iteration = 1234")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(iterationId: 1234, before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and iteration = 1234")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(iterationId: 1234, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_iteration_equals_list() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and iteration = (12, 34)"),
        "Error: `iteration` does not support the equals (`=`) operator for `(12, 34)`. Supported operators: is one of (`in`), not equals (`!=`)."
    );
}

#[test]
fn test_iteration_in_list() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and iteration in (12, 34)")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(iterationId: [12, 34], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and iteration in (12, 34)")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(iterationId: [12, 34], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_iteration_not_equals_list() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and iteration != (12, 34)")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(not: {iterationId: [12, 34]}, before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and iteration != (12, 34)")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(not: {iterationId: [12, 34]}, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_cadence_equals_value() {
    assert_eq!(
        compile_graphql("cadence = 1234").lines().nth(1).unwrap(),
        "  issues(iterationCadenceId: \"gid://gitlab/Iterations::Cadence/1234\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and iteration != (12, 34)")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(not: {iterationId: [12, 34]}, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_cadence_equals_list() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and cadence = (12, 34)"),
        "Error: `cadence` does not support the equals (`=`) operator for `(12, 34)`. Supported operators: is one of (`in`)."
    );
}

#[test]
fn test_cadence_in_list() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and cadence in (12, 34)")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(iterationCadenceId: [\"gid://gitlab/Iterations::Cadence/12\", \"gid://gitlab/Iterations::Cadence/34\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and cadence in (12, 34)")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(iterationCadenceId: [\"gid://gitlab/Iterations::Cadence/12\", \"gid://gitlab/Iterations::Cadence/34\"], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_cadence_not_equals_list() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and cadence != (12, 34)"),
        "Error: `cadence` does not support the not equals (`!=`) operator for `(12, 34)`. Supported operators: is one of (`in`)."
    );
}

#[test]
fn test_cadence_and_current_iteration() {
    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and cadence = 1234 and iteration = current"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    issues(iterationCadenceId: \"gid://gitlab/Iterations::Cadence/1234\", iterationWildcardId: CURRENT, before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and cadence = 1234 and iteration = current"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    workItems(iterationCadenceId: \"gid://gitlab/Iterations::Cadence/1234\", iterationWildcardId: CURRENT, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_iteration_with_reference_equals_value() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and iteration = *iteration:1234")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(iterationId: \"1234\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and iteration = *iteration:1234")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(iterationId: \"1234\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_invalid_iteration_equals_label() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and iteration = ~frontend"),
        "Error: `iteration` cannot be compared with `~frontend`. Supported value types: `String`, `List`, `Number`, `Nullable` (`null`, `none`, `any`), `Enum` (`current`), `Iteration` (example: `*iteration:25263`."
    )
}

#[test]
fn test_invalid_iteration_reference() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and iteration = *iteration:123foobar"),
        "Error: Invalid iteration reference `*iteration:123foobar`"
    );
}
