pub mod test_helpers;

use crate::test_helpers::*;
use glql::utils::feature_flags::FeatureFlag;

#[test]
fn test_field_selection_for_work_items() {
    FeatureFlag::GlqlWorkItems.set(true);

    let fields = vec![
        // static fields
        "author",
        "closedAt",
        "confidential",
        "createdAt",
        "description",
        "id",
        "state",
        "title",
        "updatedAt",
        "webUrl",
        "type",
        "reference",
        "project",
        // widget related fields
        "labels",
        "assignees",
        "milestone",
        "startDate",
        "dueDate",
        "healthStatus",
        "weight",
        "iteration",
        "status",
        "parent", // also available as epic
        "progress",
        "color",
        "taskCompletionStatus",
        "lastComment",
        "timeEstimate",
        "totalTimeSpent",
    ];

    let response = compile_with_fields("type = Issue and group = \"gitlab-org\"", &fields);

    fields.iter().for_each(|field| {
        assert!(response.contains(field), "{field} is not in the response");
    });
}

#[test]
fn test_field_selection_for_issues_with_aliases_with_work_items_enabled() {
    FeatureFlag::GlqlWorkItems.set(true);

    test_field_selection_for_issues_with_aliases();

    let fields = [
        ("epic", "parent"),
        ("start", "startDate"),
        ("type", "workItemType"),
    ];

    let response = compile_with_fields(
        "type = Issue and group = \"gitlab-org\"",
        &fields.iter().map(|(alias, _)| *alias).collect::<Vec<_>>(),
    );

    fields.iter().for_each(|(alias, field)| {
        assert!(
            response.contains(field),
            "{field} (alias: {alias}) is not in the response"
        );
    });
}

#[test]
fn test_field_selection_for_issues() {
    let fields = vec![
        "assignees",
        "author",
        "closedAt",
        "confidential",
        "createdAt",
        "description",
        "dueDate",
        "healthStatus",
        "id",
        "iteration",
        "labels",
        "milestone",
        "state",
        "status",
        "taskCompletionStatus",
        "timeEstimate",
        "title",
        "updatedAt",
        "webUrl",
        "weight",
        "epic",
        "reference",
        "timeEstimate",
        "totalTimeSpent",
    ];

    let response = compile_with_fields("type = Issue and group = \"gitlab-org\"", &fields);

    fields.iter().for_each(|field| {
        assert!(response.contains(field), "{field} is not in the response");
    });
}

#[test]
fn test_field_selection_for_issues_with_aliases() {
    let fields = vec![
        ("assignee", "assignees"),
        ("closed", "closedAt"),
        ("created", "createdAt"),
        ("due", "dueDate"),
        ("health", "healthStatus"),
        ("updated", "updatedAt"),
        ("description", "descriptionHtml"),
    ];

    let response = compile_with_fields(
        "type = Issue and group = \"gitlab-org\"",
        &fields.iter().map(|(alias, _)| *alias).collect::<Vec<_>>(),
    );

    fields.iter().for_each(|(alias, field)| {
        assert!(
            response.contains(field),
            "{field} (alias: {alias}) is not in the response"
        );
    });
}

#[test]
fn test_field_selection_for_epics() {
    let fields = vec![
        "author",
        "closedAt",
        "color",
        "confidential",
        "createdAt",
        "description",
        "dueDate",
        "iid",
        "labels",
        "startDate",
        "state",
        "healthStatus",
        "title",
        "updatedAt",
        "webUrl",
    ];

    let response = compile_with_fields("type = Epic and group = \"gitlab-org\"", &fields);

    fields.iter().for_each(|field| {
        assert!(response.contains(field), "{field} is not in the response");
    });
}

#[test]
fn test_field_selection_for_epics_with_aliases() {
    let fields = vec![
        ("closed", "closedAt"),
        ("created", "createdAt"),
        ("due", "dueDate"),
        ("start", "startDate"),
        ("health", "healthStatus"),
        ("updated", "updatedAt"),
        ("description", "descriptionHtml"),
    ];

    let response = compile_with_fields(
        "type = Epic and group = \"gitlab-org\"",
        &fields.iter().map(|(alias, _)| *alias).collect::<Vec<_>>(),
    );

    fields.iter().for_each(|(alias, field)| {
        assert!(
            response.contains(field),
            "{field} (alias: {alias}) is not in the response"
        );
    });
}

#[test]
fn test_field_selection_for_merge_requests() {
    let fields = vec![
        "approved",
        "approvedBy",
        "reviewers",
        "assignees",
        "author",
        "closedAt",
        "createdAt",
        "description",
        "draft",
        "iid",
        "labels",
        "milestone",
        "state",
        "title",
        "updatedAt",
        "webUrl",
        "mergedAt",
        "mergeUser",
        "reviewers",
        "sourceProject",
        "targetProject",
        "sourceBranch",
        "targetBranch",
        "state",
        "webUrl",
        "timeEstimate",
        "totalTimeSpent",
    ];

    let response = compile_with_fields("type = MergeRequest and group = \"gitlab-org\"", &fields);

    fields.iter().for_each(|field| {
        assert!(response.contains(field), "{field} is not in the response");
    });
}

#[test]
fn test_field_selection_for_merge_requests_with_aliases() {
    let fields = vec![
        ("approver", "approvedBy"),
        ("approvers", "approvedBy"),
        ("reviewer", "reviewers"),
        ("reviewedBy", "reviewers"),
        ("merged", "mergedAt"),
        ("merger", "mergeUser"),
        ("mergedBy", "mergeUser"),
        ("assignee", "assignees"),
        ("closed", "closedAt"),
        ("created", "createdAt"),
        ("updated", "updatedAt"),
        ("description", "descriptionHtml"),
    ];

    let response = compile_with_fields(
        "type = MergeRequest and group = \"gitlab-org\"",
        &fields.iter().map(|(alias, _)| *alias).collect::<Vec<_>>(),
    );

    fields.iter().for_each(|(alias, field)| {
        assert!(
            response.contains(field),
            "{field} (alias: {alias}) is not in the response"
        );
    });
}
