pub mod test_helpers;

use crate::test_helpers::*;
use glql::{
    compile_glql,
    types::{Context, Variable},
    utils::feature_flags::FeatureFlag,
};

#[test]
fn test_epic_equals_null() {
    assert_eq!(
        compile_graphql("epic = null").lines().nth(1).unwrap(),
        "  issues(epicWildcardId: NONE, before: $before, after: $after, first: $limit, includeSubepics: true) {"
    );
}

#[test]
fn test_epic_equals_none_token() {
    assert_eq!(
        compile_graphql("epic = none").lines().nth(1).unwrap(),
        "  issues(epicWildcardId: NONE, before: $before, after: $after, first: $limit, includeSubepics: true) {"
    );
}

#[test]
fn test_epic_equals_any_token() {
    assert_eq!(
        compile_graphql("epic = any").lines().nth(1).unwrap(),
        "  issues(epicWildcardId: ANY, before: $before, after: $after, first: $limit, includeSubepics: true) {"
    );
}

#[test]
fn test_epic_equals_number() {
    let mut context = Context::default();
    let compiled = compile_glql("group = \"foo\" and epic = 2", &mut context);
    assert!(
        context.variables[0].key.starts_with("epicId")
            && context.variables[0].key[6..]
                .chars()
                .all(|c| c.is_ascii_digit())
    );
    assert_eq!(
        compiled.output.lines().next().unwrap(),
        format!(
            "query GLQL(${}: String, $before: String, $after: String, $limit: Int) {{",
            context.variables[0].key
        )
    );
    assert_eq!(
        compiled.output.lines().nth(2).unwrap(),
        format!(
            "    issues(epicId: ${}, before: $before, after: $after, first: $limit, includeSubgroups: true, includeSubepics: true) {{",
            context.variables[0].key
        )
    );

    assert_eq!(
        compiled.variables[0..1],
        vec![
            Variable::new(&context.variables[0].key, "String").with_value(
                r#"query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "foo") {
    epics(iids: "2", includeDescendantGroups: false, before: $before, after: $after, first: $limit) {
      nodes {
        id
      }
    }
  }
}
"#
            )
        ],
    )
}

#[test]
fn test_epic_equals_string() {
    let mut context = Context::default();
    let compiled = compile_glql("group = \"foo\" and epic = \"2\"", &mut context);
    assert!(
        context.variables[0].key.starts_with("epicId")
            && context.variables[0].key[6..]
                .chars()
                .all(|c| c.is_ascii_digit())
    );
    assert_eq!(
        compiled.output.lines().next().unwrap(),
        format!(
            "query GLQL(${}: String, $before: String, $after: String, $limit: Int) {{",
            context.variables[0].key
        )
    );
    assert_eq!(
        compiled.output.lines().nth(2).unwrap(),
        format!(
            "    issues(epicId: ${}, before: $before, after: $after, first: $limit, includeSubgroups: true, includeSubepics: true) {{",
            context.variables[0].key
        )
    );

    assert_eq!(
        compiled.variables[0..1],
        vec![
            Variable::new(&context.variables[0].key, "String").with_value(
                r#"query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "foo") {
    epics(iids: "2", includeDescendantGroups: false, before: $before, after: $after, first: $limit) {
      nodes {
        id
      }
    }
  }
}
"#
            )
        ],
    )
}

#[test]
fn test_epic_equals_group_and_id() {
    let mut context = Context::default();

    let compiled = compile_glql("epic = gitlab-org&123", &mut context);
    assert!(
        context.variables[0].key.starts_with("epicId")
            && context.variables[0].key[6..]
                .chars()
                .all(|c| c.is_ascii_digit())
    );
    assert_eq!(
        compiled.output.lines().next().unwrap(),
        format!(
            "query GLQL(${}: String, $before: String, $after: String, $limit: Int) {{",
            context.variables[0].key
        )
    );
    assert_eq!(
        compiled.output.lines().nth(1).unwrap(),
        format!(
            "  issues(epicId: ${}, before: $before, after: $after, first: $limit, includeSubepics: true) {{",
            context.variables[0].key
        )
    );

    assert_eq!(
        compiled.variables[0..1],
        vec![
            Variable::new(&context.variables[0].key, "String").with_value(
                r#"query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "gitlab-org") {
    epics(iids: "123", includeDescendantGroups: false, before: $before, after: $after, first: $limit) {
      nodes {
        id
      }
    }
  }
}
"#
            )
        ],
    )
}

#[test]
fn test_epic_equals_string_with_work_items() {
    FeatureFlag::GlqlWorkItems.set(true);
    let mut context = Context::default();

    let compiled = compile_glql(
        "project = \"gitlab-org/gitlab\" and epic = \"gitlab-org&1\"",
        &mut context,
    );
    assert_eq!(compiled.variables[0].r#type, "WorkItemID!");
    assert!(
        context.variables[0].key.starts_with("epicId")
            && context.variables[0].key[6..]
                .chars()
                .all(|c| c.is_ascii_digit())
    );

    assert_eq!(
        compiled.output.lines().next().unwrap(),
        format!(
            "query GLQL(${}: WorkItemID!, $before: String, $after: String, $limit: Int) {{",
            context.variables[0].key
        )
    );
    assert_eq!(
        compiled.output.lines().nth(2).unwrap(),
        format!(
            "    workItems(parentIds: [${}], before: $before, after: $after, first: $limit, includeDescendantWorkItems: true) {{",
            context.variables[0].key
        )
    );

    assert_eq!(
        compiled.variables[0..1],
        vec![
            Variable::new(&context.variables[0].key, "WorkItemID!").with_value(
                r#"query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "gitlab-org") {
    workItems(types: EPIC, iids: "1", includeDescendants: false, before: $before, after: $after, first: $limit, excludeProjects: true) {
      nodes {
        id
      }
    }
  }
}
"#
            ),
        ]
    );
}

#[test]
fn test_epic_in_list_with_work_items() {
    FeatureFlag::GlqlWorkItems.set(true);
    let mut context = Context::default();

    let compiled = compile_glql(
        "project = \"gitlab-org/gitlab\" and epic in (gitlab-test&1, gitlab-test&2, gitlab-com&3, &4)",
        &mut context,
    );
    for i in 0..4 {
        assert!(
            context.variables[i].key.starts_with("epicId")
                && context.variables[i].key[6..]
                    .chars()
                    .all(|c| c.is_ascii_digit())
        );
    }

    assert_eq!(
        compiled.output.lines().next().unwrap(),
        format!(
            "query GLQL(${}: WorkItemID!, ${}: WorkItemID!, ${}: WorkItemID!, ${}: WorkItemID!, $before: String, $after: String, $limit: Int) {{",
            context.variables[0].key,
            context.variables[1].key,
            context.variables[2].key,
            context.variables[3].key
        )
    );

    assert_eq!(
        compiled.output.lines().nth(2).unwrap(),
        format!(
            "    workItems(parentIds: [${}, ${}, ${}, ${}], before: $before, after: $after, first: $limit, includeDescendantWorkItems: true) {{",
            context.variables[0].key,
            context.variables[1].key,
            context.variables[2].key,
            context.variables[3].key
        )
    );

    assert_eq!(
        compiled.variables[0..4],
        vec![
            Variable::new(&context.variables[0].key, "WorkItemID!").with_value(
                r#"query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "gitlab-test") {
    workItems(types: EPIC, iids: "1", includeDescendants: false, before: $before, after: $after, first: $limit, excludeProjects: true) {
      nodes {
        id
      }
    }
  }
}
"#
            ),
            Variable::new(&context.variables[1].key, "WorkItemID!").with_value(
                r#"query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "gitlab-test") {
    workItems(types: EPIC, iids: "2", includeDescendants: false, before: $before, after: $after, first: $limit, excludeProjects: true) {
      nodes {
        id
      }
    }
  }
}
"#
            ),
            Variable::new(&context.variables[2].key, "WorkItemID!").with_value(
                r#"query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "gitlab-com") {
    workItems(types: EPIC, iids: "3", includeDescendants: false, before: $before, after: $after, first: $limit, excludeProjects: true) {
      nodes {
        id
      }
    }
  }
}
"#
            ),
            Variable::new(&context.variables[3].key, "WorkItemID!").with_value(
                r#"query GLQL($before: String, $after: String, $limit: Int) {
  group(fullPath: "gitlab-org") {
    workItems(types: EPIC, iids: "4", includeDescendants: false, before: $before, after: $after, first: $limit, excludeProjects: true) {
      nodes {
        id
      }
    }
  }
}
"#
            ),
        ],
    );
}

#[test]
fn test_type_equals_epic() {
    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_with_group_context("gitlab-org", "type = Epic")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(types: EPIC, before: $before, after: $after, first: $limit, includeDescendants: true, excludeProjects: true) {"
    );

    FeatureFlag::GlqlWorkItems.set(false);

    assert_eq!(
        compile_with_group_context("gitlab-org", "type = Epic")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(types: EPIC, before: $before, after: $after, first: $limit, includeDescendants: true, excludeProjects: true) {"
    );
}

#[test]
fn test_type_in_issue_and_epic() {
    assert_eq!(
        compile_with_group_context("gitlab-org", "type in (Issue, Epic)"),
        "Error: Type `Epic` cannot be combined with other types (`Issue`) in the same query. Try using `type = Epic` instead."
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_with_group_context("gitlab-org", "type in (Issue, Epic)")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(types: [ISSUE, EPIC], before: $before, after: $after, first: $limit, includeDescendants: true) {"
    )
}
