pub mod test_helpers;

use crate::test_helpers::*;
use glql::utils::feature_flags::FeatureFlag;

#[test]
fn test_due_gt_absolute_date() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due > 2024-02-01")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(dueAfter: \"2024-02-01 23:59\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due > 2024-02-01")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(dueAfter: \"2024-02-01 23:59\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_due_gte_absolute_date() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due >= 2024-02-01")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(dueAfter: \"2024-02-01 00:00\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due >= 2024-02-01")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(dueAfter: \"2024-02-01 00:00\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_due_lt_absolute_date() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due < 2024-02-01")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(dueBefore: \"2024-02-01 00:00\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due < 2024-02-01")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(dueBefore: \"2024-02-01 00:00\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_due_lte_absolute_date() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due <= 2024-02-01")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(dueBefore: \"2024-02-01 23:59\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due <= 2024-02-01")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(dueBefore: \"2024-02-01 23:59\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_due_lt_absolute_date_string() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due < \"2024-02-01\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(dueBefore: \"2024-02-01 00:00\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due < \"2024-02-01\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(dueBefore: \"2024-02-01 00:00\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_due_lte_absolute_date_string() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due <= \"2024-02-01\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(dueBefore: \"2024-02-01 23:59\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due <= \"2024-02-01\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(dueBefore: \"2024-02-01 23:59\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_due_eq_absolute_date() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due = \"2024-02-01\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(dueAfter: \"2024-02-01 00:00\", dueBefore: \"2024-02-01 23:59\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due = \"2024-02-01\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(dueAfter: \"2024-02-01 00:00\", dueBefore: \"2024-02-01 23:59\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_due_gt_relative_date_days() {
    assert_eq!(
        compile_with_time_from_string(
            "2024-11-27",
            "project = \"gitlab-org/gitlab\" and due > -3d"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    issues(dueAfter: \"2024-11-24 23:59\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_with_time_from_string(
            "2024-11-27",
            "project = \"gitlab-org/gitlab\" and due > -3d"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    workItems(dueAfter: \"2024-11-24 23:59\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_due_lte_relative_date_months() {
    assert_eq!(
        compile_with_time_from_string(
            "2024-11-27",
            "project = \"gitlab-org/gitlab\" and due <= 4m"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    issues(dueBefore: \"2025-03-27 23:59\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_with_time_from_string(
            "2024-11-27",
            "project = \"gitlab-org/gitlab\" and due <= 4m"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    workItems(dueBefore: \"2025-03-27 23:59\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_due_eq_relative_date_years() {
    assert_eq!(
        compile_with_time_from_string(
            "2024-11-27",
            "project = \"gitlab-org/gitlab\" and due = -1y"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    issues(dueAfter: \"2023-11-27 00:00\", dueBefore: \"2023-11-27 23:59\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_with_time_from_string(
            "2024-11-27",
            "project = \"gitlab-org/gitlab\" and due = -1y"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    workItems(dueAfter: \"2023-11-27 00:00\", dueBefore: \"2023-11-27 23:59\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_due_gt_relative_date_weeks() {
    assert_eq!(
        compile_with_time_from_string("2024-11-27", "project = \"gitlab-org/gitlab\" and due > 2w")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(dueAfter: \"2024-12-11 23:59\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_with_time_from_string("2024-11-27", "project = \"gitlab-org/gitlab\" and due > 2w")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(dueAfter: \"2024-12-11 23:59\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_due_lt_lte_equivalence() {
    // `due < -5d` is equivalent to `due <= -6d`
    assert_eq!(
        compile_with_time_from_string(
            "2024-11-27",
            "project = \"gitlab-org/gitlab\" and due < -5d"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    issues(dueBefore: \"2024-11-22 00:00\", before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_with_time_from_string(
            "2024-11-27",
            "project = \"gitlab-org/gitlab\" and due <= -6d"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    issues(dueBefore: \"2024-11-21 23:59\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_with_time_from_string(
            "2024-11-27",
            "project = \"gitlab-org/gitlab\" and due <= -6d"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    workItems(dueBefore: \"2024-11-21 23:59\", before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_with_time_from_string(
            "2024-11-27",
            "project = \"gitlab-org/gitlab\" and due < -5d"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    workItems(dueBefore: \"2024-11-22 00:00\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_due_gt_gte_equivalanece() {
    // `due > -5d` is equivalent to `due >= -4d`
    assert_eq!(
        compile_with_time_from_string(
            "2024-11-27",
            "project = \"gitlab-org/gitlab\" and due > -5d"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    issues(dueAfter: \"2024-11-22 23:59\", before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_with_time_from_string(
            "2024-11-27",
            "project = \"gitlab-org/gitlab\" and due >= -4d"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    issues(dueAfter: \"2024-11-23 00:00\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_with_time_from_string(
            "2024-11-27",
            "project = \"gitlab-org/gitlab\" and due > -5d"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    workItems(dueAfter: \"2024-11-22 23:59\", before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_with_time_from_string(
            "2024-11-27",
            "project = \"gitlab-org/gitlab\" and due >= -4d"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    workItems(dueAfter: \"2024-11-23 00:00\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_due_invalid_operator() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due != \"2024-02-01\""),
        "Error: `due` does not support the not equals (`!=`) operator. Supported operators: equals (`=`), greater than (`>`), greater than or equal to (`>=`), less than (`<`), less than or equal to (`<=`)."
    );
}

#[test]
fn test_due_invalid_date_format() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due = \"random string\""),
        "Error: `due` cannot be compared with `\"random string\"`. Supported value types: `Date`."
    );
}

#[test]
fn test_due_invalid_number_value() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due > 12345"),
        "Error: `due` cannot be compared with `12345`. Supported value types: `Date`."
    );
}

#[test]
fn test_due_invalid_null_value() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due < null"),
        "Error: `due` cannot be compared with `null`. Supported value types: `Date`."
    );
}

#[test]
fn test_due_invalid_boolean_value() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and due = true"),
        "Error: `due` cannot be compared with `true`. Supported value types: `Date`."
    );
}
