pub mod test_helpers;
use crate::test_helpers::compile_graphql;
use glql::utils::feature_flags::FeatureFlag;

macro_rules! test {
    ($($name:ident: ($input:expr, $expected_issues:expr, $expected_epics:expr, $expected_work_items:expr),)*) => { $(
        #[test]
        fn $name() {
            // test issues
            let compiled = compile_graphql(concat!("project = \"gitlab-org/gitlab\" and ", $input));
            assert_eq!(compiled.lines().nth(2).unwrap_or_else(|| compiled.as_str()), $expected_issues);

            // test epics
            let compiled = compile_graphql(concat!("group = \"gitlab-org\" and type = Epic and ", $input));
            assert_eq!(compiled.lines().nth(2).unwrap_or_else(|| compiled.as_str()), $expected_epics);

            // test work items
            FeatureFlag::GlqlWorkItems.set(true);
            let compiled = compile_graphql(concat!("project = \"gitlab-org/gitlab\" and ", $input));
            assert_eq!(compiled.lines().nth(2).unwrap_or_else(|| compiled.as_str()), $expected_work_items);

        }
    )* }
}

test! {
    test_author_equals_string: (
        "author = \"foo\"",
        "    issues(authorUsername: \"foo\", before: $before, after: $after, first: $limit) {",
        "    workItems(types: EPIC, authorUsername: \"foo\", before: $before, after: $after, first: $limit, includeDescendants: true, excludeProjects: true) {",
        "    workItems(authorUsername: \"foo\", before: $before, after: $after, first: $limit) {"
    ),
    test_author_equals_username_reference: (
        "author = @foo",
        "    issues(authorUsername: \"foo\", before: $before, after: $after, first: $limit) {",
        "    workItems(types: EPIC, authorUsername: \"foo\", before: $before, after: $after, first: $limit, includeDescendants: true, excludeProjects: true) {",
        "    workItems(authorUsername: \"foo\", before: $before, after: $after, first: $limit) {"
    ),
    test_author_equals_array: (
        "author = (\"foo\", \"bar\")",
        "Error: `author` does not support the equals (`=`) operator for `(\"foo\", \"bar\")`. Supported operators: is one of (`in`), not equals (`!=`).",
        // epics don't support not equals array
        "Error: `author` does not support the equals (`=`) operator for `(\"foo\", \"bar\")`. Supported operators: is one of (`in`), not equals (`!=`).",
        "Error: `author` does not support the equals (`=`) operator for `(\"foo\", \"bar\")`. Supported operators: is one of (`in`), not equals (`!=`)."
    ),
    test_author_in_array: (
        "author in (\"foo\", \"bar\")",
        "    issues(or: {authorUsernames: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit) {",
        "    workItems(types: EPIC, or: {authorUsernames: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit, includeDescendants: true, excludeProjects: true) {",
        "    workItems(or: {authorUsernames: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit) {"
    ),
    test_author_in_array_of_username_references: (
        "author in (@foo, @bar)",
        "    issues(or: {authorUsernames: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit) {",
        "    workItems(types: EPIC, or: {authorUsernames: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit, includeDescendants: true, excludeProjects: true) {",
        "    workItems(or: {authorUsernames: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit) {"
    ),
    test_author_in_mixed_array: (
        "author in (\"foo\", @bar)",
        "    issues(or: {authorUsernames: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit) {",
        "    workItems(types: EPIC, or: {authorUsernames: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit, includeDescendants: true, excludeProjects: true) {",
        "    workItems(or: {authorUsernames: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit) {"
    ),
    test_author_not_equal_to_string: (
        "author != \"foo\"",
        "    issues(not: {authorUsername: \"foo\"}, before: $before, after: $after, first: $limit) {",
        "    workItems(types: EPIC, not: {authorUsername: \"foo\"}, before: $before, after: $after, first: $limit, includeDescendants: true, excludeProjects: true) {",
        "    workItems(not: {authorUsername: \"foo\"}, before: $before, after: $after, first: $limit) {"
    ),
    test_author_not_equal_to_array: (
        "author != (\"foo\", \"bar\")",
        "    issues(not: {authorUsername: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit) {",
        // epics don't support not equals array
        "    workItems(types: EPIC, not: {authorUsername: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit, includeDescendants: true, excludeProjects: true) {",
        "    workItems(not: {authorUsername: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit) {"
    ),
}
