pub mod test_helpers;

use crate::test_helpers::compile_graphql;
use glql::utils::feature_flags::FeatureFlag;

#[test]
fn test_assignee_equals_string() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee = \"foo\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(assigneeUsernames: \"foo\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee = \"foo\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(assigneeUsernames: \"foo\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_assignee_equals_username_reference() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee = @foo")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(assigneeUsernames: \"foo\", before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee = @foo")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(assigneeUsernames: \"foo\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_assignee_equals_array() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee=(\"foo\", \"bar\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(assigneeUsernames: [\"foo\", \"bar\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee=(\"foo\", \"bar\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(assigneeUsernames: [\"foo\", \"bar\"], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_assignee_equals_array_of_username_references() {
    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and assignee=(@.foo, @bar-baz, @qux_quux)"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    issues(assigneeUsernames: [\".foo\", \"bar-baz\", \"qux_quux\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and assignee=(@.foo, @bar-baz, @qux_quux)"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    workItems(assigneeUsernames: [\".foo\", \"bar-baz\", \"qux_quux\"], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_assignee_equals_mixed_array() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee=(\"foo\", @bar, \"baz\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(assigneeUsernames: [\"foo\", \"bar\", \"baz\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee=(\"foo\", @bar, \"baz\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(assigneeUsernames: [\"foo\", \"bar\", \"baz\"], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_assignee_equals_empty_array() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee=()")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(assigneeWildcardId: NONE, before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee=()")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(assigneeWildcardId: NONE, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_assignee_not_equals_empty_array() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee != ()")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(assigneeWildcardId: ANY, before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee != ()")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(assigneeWildcardId: ANY, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_assignee_in() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee in (\"foo\", \"bar\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(or: {assigneeUsernames: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee in (\"foo\", \"bar\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(or: {assigneeUsernames: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_assignee_multiple_operator() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee in (\"foo\", \"bar\") and assignee = \"baz\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(assigneeUsernames: \"baz\", or: {assigneeUsernames: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee in (\"foo\", \"bar\") and assignee = \"baz\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(assigneeUsernames: \"baz\", or: {assigneeUsernames: [\"foo\", \"bar\"]}, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_assignee_two_equals_string() {
    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and assignee = \"foo\" and assignee = \"bar\""
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    issues(assigneeUsernames: [\"foo\", \"bar\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and assignee = \"foo\" and assignee = \"bar\""
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    workItems(assigneeUsernames: [\"foo\", \"bar\"], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_assignee_multiple_equals_string() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee = \"foo\" and assignee = \"bar\" and assignee=\"boo\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(assigneeUsernames: [\"foo\", \"bar\", \"boo\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee = \"foo\" and assignee = \"bar\" and assignee=\"boo\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(assigneeUsernames: [\"foo\", \"bar\", \"boo\"], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_assignee_equals_string_array() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee = \"foo\" and assignee = (\"bar\", \"boo\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(assigneeUsernames: [\"bar\", \"boo\", \"foo\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee = \"foo\" and assignee = (\"bar\", \"boo\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(assigneeUsernames: [\"bar\", \"boo\", \"foo\"], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_assignee_multiple_equals_string_array() {
    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and assignee = \"foo\" and assignee = (\"bar\", \"boo\") and assignee=\"baz\""
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    issues(assigneeUsernames: [\"bar\", \"boo\", \"foo\", \"baz\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and assignee = \"foo\" and assignee = (\"bar\", \"boo\") and assignee=\"baz\""
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    workItems(assigneeUsernames: [\"bar\", \"boo\", \"foo\", \"baz\"], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_assignee_multiple_equals_array() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee = (\"foo\", \"bar\") and assignee=(\"boo\", \"baz\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(assigneeUsernames: [\"bar\", \"baz\", \"boo\", \"foo\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee = (\"foo\", \"bar\") and assignee=(\"boo\", \"baz\")")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(assigneeUsernames: [\"bar\", \"baz\", \"boo\", \"foo\"], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_assignee_not_equals() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee != \"foo\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(not: {assigneeUsernames: \"foo\"}, before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee != \"foo\"")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(not: {assigneeUsernames: \"foo\"}, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_assignee_with_null() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee = null")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(assigneeId: \"NONE\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_assignee_with_id() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee=123")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(assigneeId: \"123\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_assignee_equals_milestone_reference() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee = %Backlog"),
        "Error: `assignee` cannot be compared with `%Backlog`. Supported value types: `String`, `Username` (example: `@username`), `List`, `Number`, `Nullable` (`null`, `none`, `any`)."
    );
}

#[test]
fn test_assignee_equals_mixed_references() {
    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and assignee = @foo and assignee = (@bar, @baz)"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    issues(assigneeUsernames: [\"bar\", \"baz\", \"foo\"], before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee = @foo and assignee = @bar")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(assigneeUsernames: [\"foo\", \"bar\"], before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and assignee = \"foo\" and assignee = (@bar, @baz)"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    issues(assigneeUsernames: [\"bar\", \"baz\", \"foo\"], before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee = \"foo\" and assignee = (\"bar\", @baz) and assignee = @quox")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(assigneeUsernames: [\"bar\", \"baz\", \"foo\", \"quox\"], before: $before, after: $after, first: $limit) {"
    );

    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee = \"foo\" and assignee = (\"bar\", @baz) and assignee = @quox")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(assigneeUsernames: [\"bar\", \"baz\", \"foo\", \"quox\"], before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee = @foo and assignee = @bar")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(assigneeUsernames: [\"foo\", \"bar\"], before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_graphql(
            "project = \"gitlab-org/gitlab\" and assignee = \"foo\" and assignee = (@bar, @baz)"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    workItems(assigneeUsernames: [\"bar\", \"baz\", \"foo\"], before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and assignee = \"foo\" and assignee = (\"bar\", @baz) and assignee = @quox")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(assigneeUsernames: [\"bar\", \"baz\", \"foo\", \"quox\"], before: $before, after: $after, first: $limit) {"
    );
}
