use lazy_static::lazy_static;
use serde::Deserialize;
use std::cell::RefCell;
use std::collections::HashMap;

#[derive(Debug, Clone, Copy, Hash, Eq, PartialEq)]
pub enum FeatureFlag {
    GlqlWorkItems,
}

#[derive(Deserialize)]
#[serde(rename_all = "camelCase")]
pub struct FeatureFlags {
    pub glql_work_items: Option<bool>,
}

lazy_static! {
    static ref DEFAULT_FLAGS: HashMap<FeatureFlag, bool> = {
        let mut map = HashMap::new();
        map.insert(FeatureFlag::GlqlWorkItems, false);
        map
    };
}

thread_local! {
    static FEATURE_FLAGS: RefCell<HashMap<FeatureFlag, bool>> = RefCell::new({
        let mut map = HashMap::new();
        for (flag, default_value) in DEFAULT_FLAGS.iter() {
            map.insert(*flag, *default_value);
        }
        map
    });
}

impl FeatureFlag {
    /// Get the current status of a feature flag.
    pub fn get(self) -> bool {
        FEATURE_FLAGS.with(|flags| *flags.borrow().get(&self).unwrap_or(&false))
    }

    /// Set the value of a feature flag.
    pub fn set(self, value: bool) {
        FEATURE_FLAGS.with(|flags| {
            flags.borrow_mut().insert(self, value);
        });
    }

    /// Reset the value of a feature flag to its default state.
    pub fn reset(self) {
        FEATURE_FLAGS.with(|flags| {
            flags
                .borrow_mut()
                .insert(self, *DEFAULT_FLAGS.get(&self).unwrap_or(&false));
        });
    }

    /// Reset all feature flags to their default states.
    pub fn reset_all() {
        for flag in DEFAULT_FLAGS.keys() {
            flag.reset();
        }
    }
}
