use crate::utils::date_time::{TimeSegmentType, TimeUnit, time_segment};
use chrono::{DateTime, Duration, NaiveDate, NaiveDateTime, Utc};

#[derive(Debug, Clone, PartialEq)]
pub struct TimeRange {
    pub from: DateTime<Utc>,
    pub to: DateTime<Utc>,
}

impl TimeRange {
    pub fn new(before: DateTime<Utc>, after: DateTime<Utc>) -> Self {
        Self {
            from: before,
            to: after,
        }
    }

    pub fn duration(&self) -> Duration {
        self.to.signed_duration_since(self.from)
    }

    pub fn from_str(from: &str, to: &str) -> Result<Self, String> {
        let parse_datetime = |s: &str| -> Result<DateTime<Utc>, String> {
            // Try parsing as DateTime first (YYYY-MM-DD HH:MM)
            if let Ok(naive) = NaiveDateTime::parse_from_str(s, "%Y-%m-%d %H:%M") {
                return Ok(naive.and_utc());
            }
            // Try as date only (YYYY-MM-DD)
            if let Ok(naive) = NaiveDate::parse_from_str(s, "%Y-%m-%d") {
                return Ok(naive.and_hms_opt(0, 0, 0).unwrap().and_utc());
            }
            Err(format!("Unable to parse datetime: {s}"))
        };

        let from = parse_datetime(from)?;
        let to = parse_datetime(to)?;
        Ok(Self { from, to })
    }

    /// Parse a key in the format "from_YYYY_MM_DD_to_YYYY_MM_DD"
    pub fn from_key(key: &str) -> Result<Self, String> {
        let (from_part, to_part) = key
            .split_once("_to_")
            .ok_or_else(|| "Invalid key".to_string())?;
        let from_str = from_part
            .strip_prefix("from_")
            .ok_or_else(|| "Invalid key".to_string())?;

        let parse_date = |s: &str| -> Result<DateTime<Utc>, String> {
            let date =
                NaiveDate::parse_from_str(s, "%Y_%m_%d").map_err(|_| "Invalid date".to_string())?;
            Ok(date.and_hms_opt(0, 0, 0).unwrap().and_utc())
        };

        Ok(Self {
            from: parse_date(from_str).map_err(|_| "Invalid from date".to_string())?,
            to: parse_date(to_part).map_err(|_| "Invalid to date".to_string())?,
        })
    }

    pub fn segment(
        &self,
        quantity: i64,
        unit: TimeUnit,
        segment_type: TimeSegmentType,
    ) -> Result<Vec<Self>, String> {
        time_segment(self, quantity, unit, segment_type)
    }

    /// Format the time range in the format "from_YYYY_MM_DD_to_YYYY_MM_DD"
    pub fn key(&self) -> String {
        format!(
            "from_{}_to_{}",
            self.from.format("%Y_%m_%d"),
            self.to.format("%Y_%m_%d")
        )
    }

    pub fn stringify(&self) -> String {
        format!(
            "From {} to {}",
            self.from.format("%Y-%m-%d"),
            self.to.format("%Y-%m-%d")
        )
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use chrono::TimeZone;

    #[test]
    fn test_from_key_valid() {
        let tr = TimeRange::from_key("from_2023_01_01_to_2023_01_31").unwrap();
        assert_eq!(tr.from, Utc.with_ymd_and_hms(2023, 1, 1, 0, 0, 0).unwrap());
        assert_eq!(tr.to, Utc.with_ymd_and_hms(2023, 1, 31, 0, 0, 0).unwrap());
    }

    #[test]
    fn test_from_key_invalid_format() {
        let err = TimeRange::from_key("2023_01_01_to_2023_01_31").unwrap_err();
        assert_eq!(err, "Invalid key");
    }

    #[test]
    fn test_from_key_invalid_dates() {
        let err_from = TimeRange::from_key("from_2023_13_01_to_2023_01_31").unwrap_err();
        assert_eq!(err_from, "Invalid from date");

        let err_to = TimeRange::from_key("from_2023_01_01_to_2023_02_30").unwrap_err();
        assert_eq!(err_to, "Invalid to date");
    }
}
