mod time_range;
mod time_segment_type;
mod time_unit;

pub use time_range::TimeRange;
pub use time_segment_type::TimeSegmentType;
pub use time_unit::TimeUnit;

use chrono::{DateTime, Datelike, Duration, NaiveDate, Timelike, Utc};
use time_segment_type::TimeSegmentType::*;
use time_unit::TimeUnit::*;

pub fn start_of_day(days: i64, time: DateTime<Utc>) -> String {
    (time + Duration::days(days))
        .date_naive()
        .format("%Y-%m-%d")
        .to_string()
}

pub fn calculate_days(quantity: i64, unit: TimeUnit, time: DateTime<Utc>) -> Result<i64, String> {
    Ok(match unit {
        Days => quantity,
        Weeks => 7 * quantity,
        Months => months_to_days(quantity, time)?,
        Years => months_to_days(quantity * 12, time)?,
    })
}

// number of days in a given number of months
// adjusts for days in the current month that have passed
pub fn months_to_days(quantity: i64, time: DateTime<Utc>) -> Result<i64, String> {
    let start_date = time.date_naive();
    let end_date = add_months(start_date, quantity)?;

    // Calculate the difference in days
    let duration = end_date.signed_duration_since(start_date);
    Ok(duration.num_days())
}

fn add_months(date: NaiveDate, quantity: i64) -> Result<NaiveDate, String> {
    let year = (date.year() as i64) + ((date.month() as i64) + quantity - 1).div_euclid(12);
    let month = ((date.month() as i64) + quantity - 1).rem_euclid(12) + 1;
    let invalid_date = || "Invalid date calculation".to_string();
    let max_day = NaiveDate::from_ymd_opt(year as i32, month as u32, 1)
        .ok_or_else(invalid_date)?
        .with_day(1)
        .ok_or_else(invalid_date)?
        .checked_add_months(chrono::Months::new(1))
        .ok_or_else(invalid_date)?
        .pred_opt()
        .ok_or_else(invalid_date)?
        .day();
    let day = std::cmp::min(date.day(), max_day);

    NaiveDate::from_ymd_opt(year as i32, month as u32, day).ok_or_else(invalid_date)
}

/// Get the next start of the specified time unit after a given datetime
fn next_start_of_unit(datetime: DateTime<Utc>, unit: TimeUnit) -> DateTime<Utc> {
    match unit {
        Days => {
            // Next day at midnight
            (datetime + Duration::days(1))
                .date_naive()
                .and_hms_opt(0, 0, 0)
                .unwrap()
                .and_utc()
        }
        Weeks => {
            // Next Monday
            let weekday = datetime.weekday();
            let days_to_next_monday = (7 - weekday.num_days_from_monday() as i64) % 7;
            let next_monday = if days_to_next_monday == 0 {
                datetime + Duration::days(7)
            } else {
                datetime + Duration::days(days_to_next_monday)
            };
            next_monday
                .date_naive()
                .and_hms_opt(0, 0, 0)
                .unwrap()
                .and_utc()
        }
        Months => {
            // Next month 1st
            let next_month = datetime
                .date_naive()
                .checked_add_months(chrono::Months::new(1))
                .unwrap_or_else(|| datetime.date_naive());
            next_month
                .with_day(1)
                .unwrap()
                .and_hms_opt(0, 0, 0)
                .unwrap()
                .and_utc()
        }
        Years => {
            // Next year January 1st
            let next_year = datetime
                .date_naive()
                .with_year(datetime.year() + 1)
                .unwrap_or_else(|| datetime.date_naive());
            next_year
                .with_month(1)
                .unwrap()
                .with_day(1)
                .unwrap()
                .and_hms_opt(0, 0, 0)
                .unwrap()
                .and_utc()
        }
    }
}

fn is_start_of_unit(datetime: DateTime<Utc>, unit: TimeUnit) -> bool {
    match unit {
        Days => datetime.hour() == 0 && datetime.minute() == 0 && datetime.second() == 0,
        Weeks => {
            datetime.weekday().num_days_from_monday() == 0
                && datetime.hour() == 0
                && datetime.minute() == 0
                && datetime.second() == 0
        }
        Months => {
            datetime.day() == 1
                && datetime.hour() == 0
                && datetime.minute() == 0
                && datetime.second() == 0
        }
        Years => {
            datetime.month() == 1
                && datetime.day() == 1
                && datetime.hour() == 0
                && datetime.minute() == 0
                && datetime.second() == 0
        }
    }
}

// Quantity-aware alignment helpers for FromStartOfUnit
fn is_aligned_to_unit(datetime: DateTime<Utc>, unit: TimeUnit, quantity: i64) -> bool {
    match unit {
        Days => is_start_of_unit(datetime, Days),
        Weeks => is_start_of_unit(datetime, Weeks),
        Months => {
            if !is_start_of_unit(datetime, Months) {
                return false;
            }
            let month_zero_indexed = (datetime.month() as i64) - 1; // Jan = 0
            (month_zero_indexed % quantity.rem_euclid(12).max(1)) == 0
        }
        Years => is_start_of_unit(datetime, Years),
    }
}

fn next_aligned_start(datetime: DateTime<Utc>, unit: TimeUnit, quantity: i64) -> DateTime<Utc> {
    match unit {
        Days => next_start_of_unit(datetime, Days),
        Weeks => next_start_of_unit(datetime, Weeks),
        Months => {
            let date = datetime.date_naive();
            let current_month_zero_indexed = (date.month() as i64) - 1; // Jan = 0
            let q = quantity.rem_euclid(12).max(1);
            let remainder = current_month_zero_indexed % q;
            let months_to_add = if remainder == 0 { q } else { q - remainder };
            let target_first_of_month = date
                .checked_add_months(chrono::Months::new(months_to_add as u32))
                .unwrap_or(date)
                .with_day(1)
                .unwrap();
            target_first_of_month
                .and_hms_opt(0, 0, 0)
                .unwrap()
                .and_utc()
        }
        Years => next_start_of_unit(datetime, Years),
    }
}

/// Divides a time range into segments based on the specified quantity and time unit.
/// Returns a vector of TimeRange segments that cover the entire input range.
pub fn time_segment(
    time_range: &TimeRange,
    quantity: i64,
    unit: TimeUnit,
    segment_type: TimeSegmentType,
) -> Result<Vec<TimeRange>, String> {
    if quantity <= 0 {
        return Err("Quantity must be positive".to_string());
    }

    let total_duration = time_range.duration();
    let total_days = total_duration.num_days();

    if total_days < 0 {
        return Err("Invalid time range: after date is before before date".to_string());
    }

    match segment_type {
        FromStartOfRange => {
            // Original logic: segment from the start of the range
            let mut segments = Vec::new();
            let mut current_start = time_range.from;

            while current_start < time_range.to {
                // Calculate the end of the current segment
                let segment_days = calculate_days(quantity, unit.clone(), current_start)?;
                let segment_duration = Duration::days(segment_days);

                let current_end = if current_start + segment_duration > time_range.to {
                    time_range.to
                } else {
                    current_start + segment_duration
                };

                segments.push(TimeRange::new(current_start, current_end));
                current_start = current_end;
            }

            Ok(segments)
        }
        FromStartOfUnit => {
            // Align to the start of units (quantity-aware), then use the requested quantity
            let mut segments = Vec::new();
            let mut current_start = time_range.from;

            while current_start < time_range.to {
                if is_aligned_to_unit(current_start, unit.clone(), quantity) {
                    // We are aligned: create a full segment of the requested quantity
                    let segment_days = calculate_days(quantity, unit.clone(), current_start)?;
                    let segment_duration = Duration::days(segment_days);
                    let desired_end = current_start + segment_duration;
                    let actual_end = if desired_end > time_range.to {
                        time_range.to
                    } else {
                        desired_end
                    };
                    segments.push(TimeRange::new(current_start, actual_end));
                    current_start = actual_end;
                } else {
                    // Not aligned: first segment goes up to the next aligned boundary
                    let next_unit_start = next_aligned_start(current_start, unit.clone(), quantity);
                    let actual_end = if next_unit_start > time_range.to {
                        time_range.to
                    } else {
                        next_unit_start
                    };
                    segments.push(TimeRange::new(current_start, actual_end));
                    current_start = actual_end;
                }
            }

            Ok(segments)
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use chrono::{NaiveDate, TimeZone, Utc};

    #[test]
    fn test_start_of_day() {
        let ctx = Utc.with_ymd_and_hms(2023, 5, 15, 12, 0, 0).unwrap();
        let result = start_of_day(0, ctx);
        assert_eq!(result, "2023-05-15");
    }

    #[test]
    fn test_calculate_days() {
        macro_rules! test_cases {
            (
                $(
                    $name:ident: (
                        $current_date:expr,
                        $quantity:expr,
                        $time_unit:expr
                    ) => $expected:expr
                )*
            ) => {
                $(
                    fn $name() {
                        let (y, m, d) = $current_date;
                        let ctx = Utc.with_ymd_and_hms(y, m, d, 0, 0, 0).unwrap();

                        let result = calculate_days($quantity, $time_unit, ctx).unwrap();
                        assert_eq!(result, $expected);
                    }

                    $name();
                )*
            };
        }

        test_cases! {
            test_days: ((2023, 1, 1), 1, Days) => 1
            test_days_negative: ((2023, 1, 1), -1, Days) => -1
            test_weeks: ((2023, 1, 1), 1, Weeks) => 7
            test_weeks_negative: ((2023, 1, 1), -2, Weeks) => -14
            test_months_jan_to_feb: ((2023, 1, 1), 1, Months) => 31
            test_months_jan_to_mar: ((2023, 1, 1), 2, Months) => 59
            test_months_full_non_leap: ((2023, 1, 1,), 12, Months) => 365
            test_years: ((2023, 1, 1), 1, Years) => 365
            test_years_negative: ((2023, 1, 1), -1, Years) => -365

            // Test for a Leap Year (2024)
            test_months_leap_year: ((2024, 1, 1), 1, Months) => 31
            test_months_leap_year_feb: ((2024, 1, 1), 2, Months) => 60
            test_months_full_leap: ((2024, 1, 1), 12, Months) => 366
            test_years_leap: ((2024, 1, 1), 1, Years) => 366
            test_years_negative_leap: ((2024, 1, 1), -1, Years) => -365

            // Test across year boundaries
            test_across_year_boundary: ((2023, 12, 15), 2, Months) => 62
            test_across_year_boundary_negative: ((2024, 2, 15), -2, Months) => -62

            // Test with larger intervals
            test_multiple_years: ((2023, 12, 15), 4, Years) => 365 * 3 + 366 // 3 normal years + 1 leap year
            test_century: ((2023, 12, 15), 100, Years) => 36524 // Approximately accounts for leap years over a century
        }
    }

    #[test]
    fn test_months_to_days() {
        macro_rules! test_cases {
            ($($name:ident: ($mock_date:expr, $months:expr) => $expected:expr)*) => {
                $(
                    fn $name(){
                        let (y, m, d) = $mock_date;
                        let ctx = Utc.with_ymd_and_hms(y, m, d, 0, 0, 0).unwrap();
                        assert_eq!(months_to_days($months, ctx).unwrap(), $expected);
                    }

                    $name();
                )*
            };
        }

        test_cases! {
            test_months_one: ((2023, 1, 1), 1) => 31
            test_months_full_year: ((2023, 1, 1), 12) => 365

            // test for leap year
            test_months_leap_one: ((2024, 1, 1), 1) => 31 // Jan to Feb in leap year
            test_months_leap_two: ((2024, 1, 1), 2) => 60 // Jan to Mar in leap year (including Feb 29)
            test_months_leap_full_year: ((2024, 1, 1), 12) => 366 // Jan to next Jan (leap year)

            // test across leap year boundary
            test_across_leap_year: ((2023, 12, 1), 3) => 91 // Dec 2023 to Mar 2024 (including Feb 29)
        }
    }

    macro_rules! add_months_test_cases {
        ($($name:ident: ($mock_date:expr, $months:expr) => $expected:expr)*) => {
        $(
            fn $name(){
                let (y, m , d) = $mock_date;
                let date = NaiveDate::from_ymd_opt(y, m, d).unwrap();

                let (y, m, d) = $expected;
                assert_eq!(
                    add_months(date, $months).unwrap(),
                    NaiveDate::from_ymd_opt(y, m, d).unwrap()
                );
            }

            $name();
        )*
        };
    }

    #[test]
    fn test_add_months() {
        add_months_test_cases! {
            test_add_one_month: ((2023, 1, 31), 1) => (2023, 2, 28)
            test_add_twelve_month: ((2023, 1, 31), 12) => (2024, 1, 31)
            test_add_to_a_leap_year: ((2024, 1, 31), 1) => (2024, 2, 29)
        }
    }

    #[test]
    fn test_add_months_negative() {
        add_months_test_cases! {
            test_subtract_one_month: ((2023, 1, 31), -1) => (2022, 12, 31)
            test_subtract_twelve_months: ((2023, 1, 31), -12) => (2022, 1, 31)
            test_subtract_thirteen_months: ((2023, 1, 31), -13) => (2021, 12, 31)
            test_subtract_large_negative: ((2023, 1, 31), -100) => (2014, 9, 30)
            test_subtract_month_with_fewer_days: ((2023, 3, 31), -1) => (2023, 2, 28)
            test_subtract_across_a_leap_year: ((2024, 3, 31), -13) => (2023, 2, 28)
        }
    }

    macro_rules! time_segment_test_cases {
        ($($name:ident: ($from:expr, $to:expr, $quantity:expr, $unit:expr, $segment_type:expr) => [$($expected_from:expr, $expected_to:expr),*])*) => {
            $(
                #[test]
                fn $name() {
                    let time_range = TimeRange::from_str($from, $to).unwrap();
                    let segments = time_segment(&time_range, $quantity, $unit, $segment_type).unwrap();

                    assert_eq!(segments, vec![
                        $(TimeRange::from_str($expected_from, $expected_to).unwrap(),)*
                    ]);
                }
            )*
        };
    }

    time_segment_test_cases! {
        test_time_segment_days: ("2023-01-01", "2023-01-05", 2, Days, FromStartOfRange) => [
            "2023-01-01", "2023-01-03",
            "2023-01-03", "2023-01-05"
        ]
        test_time_segment_weeks: ("2023-01-01", "2023-01-15", 1, Weeks, FromStartOfRange) => [
            "2023-01-01", "2023-01-08",
            "2023-01-08", "2023-01-15"
        ]
        test_time_segment_months: ("2023-01-01", "2023-03-01", 1, Months, FromStartOfRange) => [
            "2023-01-01", "2023-02-01",
            "2023-02-01", "2023-03-01"
        ]
        test_time_segment_years: ("2023-01-01", "2025-01-01", 1, Years, FromStartOfRange) => [
            "2023-01-01", "2024-01-01",
            "2024-01-01", "2025-01-01"
        ]
        test_time_segment_leap_year: ("2024-01-01", "2024-03-01", 1, Months, FromStartOfRange) => [
            "2024-01-01", "2024-02-01",
            "2024-02-01", "2024-03-01"
        ]
        test_time_segment_exact_fit: ("2023-01-01", "2023-01-03", 1, Days, FromStartOfRange) => [
            "2023-01-01", "2023-01-02",
            "2023-01-02", "2023-01-03"
        ]
        test_time_segment_smaller_than_segment: ("2023-01-01", "2023-01-02 12:00", 2, Days, FromStartOfRange) => [
            "2023-01-01", "2023-01-02 12:00"
        ]
        test_time_segment_with_time_components: ("2023-01-01 10:30", "2023-01-03 15:20", 1, Days, FromStartOfRange) => [
            "2023-01-01 10:30", "2023-01-02 10:30",
            "2023-01-02 10:30", "2023-01-03 10:30",
            "2023-01-03 10:30", "2023-01-03 15:20"
        ]
        test_time_segment_months_arbitrary_days: ("2025-01-20", "2025-04-15", 1, Months, FromStartOfRange) => [
            "2025-01-20", "2025-02-20",
            "2025-02-20", "2025-03-20",
            "2025-03-20", "2025-04-15"
        ]
        test_time_segment_months_arbitrary_days_short_range: ("2025-01-15", "2025-02-10", 1, Months, FromStartOfRange) => [
            "2025-01-15", "2025-02-10"
        ]
        test_time_segment_months_arbitrary_days_leap_year: ("2024-01-15", "2024-03-15", 1, Months, FromStartOfRange) => [
            "2024-01-15", "2024-02-15",
            "2024-02-15", "2024-03-15"
        ]
    }

    // Tests for FromStartOfUnit functionality
    time_segment_test_cases! {
        test_from_start_of_unit_months: ("2025-01-20", "2025-03-15", 1, Months, FromStartOfUnit) => [
            "2025-01-20", "2025-02-01",
            "2025-02-01", "2025-03-01",
            "2025-03-01", "2025-03-15"
        ]
        test_from_start_of_unit_months_2: ("2025-01-01", "2025-04-01", 1, Months, FromStartOfUnit) => [
            "2025-01-01", "2025-02-01",
            "2025-02-01", "2025-03-01",
            "2025-03-01", "2025-04-01"
        ]
        test_from_start_of_unit_quarterly: ("2025-01-15", "2025-12-31", 3, Months, FromStartOfUnit) => [
            "2025-01-15", "2025-04-01",
            "2025-04-01", "2025-07-01",
            "2025-07-01", "2025-10-01",
            "2025-10-01", "2025-12-31"
        ]
        test_from_start_of_unit_days: ("2025-01-20 10:30", "2025-01-22 15:45", 1, Days, FromStartOfUnit) => [
            "2025-01-20 10:30", "2025-01-21 00:00",
            "2025-01-21 00:00", "2025-01-22 00:00",
            "2025-01-22 00:00", "2025-01-22 15:45"
        ]
        test_from_start_of_unit_days_2: ("2025-01-01", "2025-01-03", 1, Days, FromStartOfUnit) => [
            "2025-01-01", "2025-01-02",
            "2025-01-02", "2025-01-03"
        ]
        test_from_start_of_unit_days_3: ("2025-01-01", "2025-01-06", 2, Days, FromStartOfUnit) => [
            "2025-01-01", "2025-01-03",
            "2025-01-03", "2025-01-05",
            "2025-01-05", "2025-01-06"
        ]
        test_from_start_of_unit_weeks: ("2025-01-18", "2025-02-03", 1, Weeks, FromStartOfUnit) => [
            "2025-01-18", "2025-01-20", // 20th is a Monday
            "2025-01-20", "2025-01-27", // 27th is a Monday
            "2025-01-27", "2025-02-03" // 3rd is a Monday
        ]
        test_from_start_of_unit_weeks_2: ("2025-01-20", "2025-02-03", 1, Weeks, FromStartOfUnit) => [
            "2025-01-20", "2025-01-27", // 20th and 27th are Mondays
            "2025-01-27", "2025-02-03" // 3rd is a Monday
        ]
        test_from_start_of_unit_weeks_3: ("2025-01-20", "2025-02-18", 2, Weeks, FromStartOfUnit) => [
            "2025-01-20", "2025-02-03",
            "2025-02-03", "2025-02-17",
            "2025-02-17", "2025-02-18"
        ]
        test_from_start_of_unit_years: ("2025-06-15", "2027-03-10", 1, Years, FromStartOfUnit) => [
            "2025-06-15", "2026-01-01",
            "2026-01-01", "2027-01-01",
            "2027-01-01", "2027-03-10"
        ]
        test_from_start_of_unit_years_2: ("2025-01-01", "2027-01-01", 1, Years, FromStartOfUnit) => [
            "2025-01-01", "2026-01-01",
            "2026-01-01", "2027-01-01"
        ]
    }

    #[test]
    fn test_time_segment_invalid_quantity() {
        let time_range = TimeRange::from_str("2023-01-01", "2023-01-05").unwrap();

        let result = time_segment(&time_range, 0, Days, FromStartOfRange);
        assert!(result.is_err());
        assert_eq!(result.unwrap_err(), "Quantity must be positive");

        let result = time_segment(&time_range, -1, Days, FromStartOfRange);
        assert!(result.is_err());
        assert_eq!(result.unwrap_err(), "Quantity must be positive");
    }

    #[test]
    fn test_time_segment_invalid_range() {
        let time_range = TimeRange::from_str("2023-01-05", "2023-01-01").unwrap();

        let result = time_segment(&time_range, 1, Days, FromStartOfRange);
        assert!(result.is_err());
        assert_eq!(
            result.unwrap_err(),
            "Invalid time range: after date is before before date"
        );
    }
}
