use lazy_static::lazy_static;
use regex::Regex;
use sha2::{Digest, Sha256};
use std::collections::HashMap;
use std::sync::Mutex;

lazy_static! {
    static ref COUNTERS: Mutex<HashMap<String, usize>> = Mutex::new(HashMap::new());
}

pub fn reset_unique_id_counters() {
    COUNTERS.lock().unwrap().clear();
}

pub fn unique_id(key: &str) -> String {
    let mut counters = COUNTERS.lock().unwrap();
    let counter = counters.entry(key.to_string()).or_insert(0);
    *counter += 1;
    format!("{key}{counter}")
}

pub fn wildcard_match(str: &str, pattern: &str) -> bool {
    let lower_pattern = pattern.to_lowercase();
    let lower_str = str.to_lowercase();

    if !lower_pattern.contains('*') {
        return lower_pattern == lower_str;
    }

    // Escape regex special characters first
    let escaped_pattern = lower_pattern
        .chars()
        .map(|c| match c {
            '\\' | '.' | '+' | '?' | '^' | '$' | '{' | '}' | '(' | ')' | '|' | '[' | ']' => {
                format!("\\{c}")
            }
            _ => c.to_string(),
        })
        .collect::<String>();

    // Then replace literal asterisks with .*
    let regex_pattern = format!("^{}$", escaped_pattern.replace("*", ".*"));

    match Regex::new(&regex_pattern) {
        Ok(re) => re.is_match(&lower_str),
        Err(_) => false,
    }
}

pub fn to_sentence_case(input: &str) -> String {
    use heck::ToTitleCase;

    if input.to_lowercase() == "id" || input.to_lowercase() == "iid" || input.len() <= 1 {
        return input.to_uppercase();
    }

    let output = input.to_title_case().to_lowercase();
    output[..1].to_uppercase() + &output[1..]
}

/// Parses an epic reference string and returns (group_path, id)
///
/// Epic references can be in the format:
/// - "123" -> uses fallback_group_path
/// - "&456" -> uses fallback_group_path
/// - "group/path&789" -> uses "group/path"
///
/// This will be used for parsing epic references with "&" prefix.
/// A similar function `parse_issue_reference` can be added later for parent fields with "#" prefix.
pub fn parse_epic_reference(reference: &str, fallback_group_path: &str) -> (String, String) {
    // Remove leading "&" if present (epic references use & prefix)
    let cleaned = reference.strip_prefix('&').unwrap_or(reference);

    // Split on & and check if we have multiple parts
    if !cleaned.contains('&') {
        return (fallback_group_path.to_string(), cleaned.to_string());
    }

    let parts: Vec<&str> = cleaned.split('&').collect();
    if parts.len() > 1 {
        let path = parts[0];
        let id = parts[1];

        (
            if path.is_empty() {
                fallback_group_path.to_string()
            } else {
                path.to_string()
            },
            id.to_string(),
        )
    } else {
        (fallback_group_path.to_string(), cleaned.to_string())
    }
}

/// Computes the SHA256 hash of a string input and returns it as a hexadecimal string
pub fn sha256(input: &str) -> String {
    let mut hasher = Sha256::new();
    hasher.update(input.as_bytes());
    format!("{:x}", hasher.finalize())
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_unique_id() {
        let base = "test";
        assert_eq!(unique_id(base), "test1");
        assert_eq!(unique_id(base), "test2");
        assert_eq!(unique_id(base), "test3");
    }

    #[test]
    fn test_to_sentence_case() {
        // Test empty string
        assert_eq!(to_sentence_case(""), "");

        // Test single character
        assert_eq!(to_sentence_case("a"), "A");
        assert_eq!(to_sentence_case("A"), "A");

        // Test special cases for "id" and "iid"
        assert_eq!(to_sentence_case("id"), "ID");
        assert_eq!(to_sentence_case("iid"), "IID");
        assert_eq!(to_sentence_case("Id"), "ID");
        assert_eq!(to_sentence_case("IID"), "IID");

        // Test simple words
        assert_eq!(to_sentence_case("hello"), "Hello");
        assert_eq!(to_sentence_case("world"), "World");
        assert_eq!(to_sentence_case("HELLO"), "Hello");
        assert_eq!(to_sentence_case("WORLD"), "World");

        // Test multiple words
        assert_eq!(to_sentence_case("hello world"), "Hello world");
        assert_eq!(to_sentence_case("HELLO WORLD"), "Hello world");
        assert_eq!(to_sentence_case("hello_world"), "Hello world");
        assert_eq!(to_sentence_case("hello-world"), "Hello world");

        // Test camelCase and PascalCase
        assert_eq!(to_sentence_case("camelCase"), "Camel case");
        assert_eq!(to_sentence_case("PascalCase"), "Pascal case");
        assert_eq!(to_sentence_case("snake_case"), "Snake case");
        assert_eq!(to_sentence_case("kebab-case"), "Kebab case");

        // Test with numbers
        assert_eq!(to_sentence_case("user123"), "User123");
        assert_eq!(to_sentence_case("123user"), "123user");
        assert_eq!(to_sentence_case("user_123_name"), "User 123 name");

        // Test with special characters
        assert_eq!(to_sentence_case("user@example.com"), "User example com");
        assert_eq!(to_sentence_case("file-name.txt"), "File name txt");
    }

    #[test]
    fn test_parse_epic_reference() {
        let fallback_group = "default-group";

        // Simple ID without group path
        assert_eq!(
            parse_epic_reference("123", fallback_group),
            ("default-group".to_string(), "123".to_string())
        );

        // ID with & prefix but no group path
        assert_eq!(
            parse_epic_reference("&456", fallback_group),
            ("default-group".to_string(), "456".to_string())
        );

        // Group path with ID
        assert_eq!(
            parse_epic_reference("gitlab-org&789", fallback_group),
            ("gitlab-org".to_string(), "789".to_string())
        );

        // Empty group path (should use fallback)
        assert_eq!(
            parse_epic_reference("&101", fallback_group),
            ("default-group".to_string(), "101".to_string())
        );

        // Complex group path
        assert_eq!(
            parse_epic_reference("gitlab-org/gitlab&202", fallback_group),
            ("gitlab-org/gitlab".to_string(), "202".to_string())
        );

        // Edge case: empty string
        assert_eq!(
            parse_epic_reference("", fallback_group),
            ("default-group".to_string(), "".to_string())
        );
    }

    #[test]
    fn test_wildcard_match() {
        // Exact matches
        assert!(wildcard_match("label", "label"));
        assert!(!wildcard_match("label", "a-label"));
        assert!(!wildcard_match("label", "label-a"));

        // Prefix wildcard (*label)
        assert!(wildcard_match("a-label", "*label"));
        assert!(!wildcard_match("a-label-a", "*label"));

        // Suffix wildcard (label*)
        assert!(wildcard_match("label-a", "label*"));
        assert!(!wildcard_match("a-label-a", "label*"));

        // Both prefix and suffix wildcard (*label*)
        assert!(wildcard_match("a-label-a", "*label*"));

        // Middle wildcard (l*l)
        assert!(wildcard_match("label", "l*l"));

        // Special characters
        assert!(wildcard_match("!@#$%^&*()-=+/?[]{}", "!@#$%^&*()-=+/?[]{}"));
    }
}
