use crate::types::{Context, ReferenceType, Variable};
use crate::utils::date_time::TimeUnit;
use Value::*;
use serde::{Deserialize, Serialize};
use std::fmt;

#[derive(Debug, PartialEq, Clone, Serialize, Deserialize, Ord, PartialOrd, Eq)]
pub enum Value {
    Null,
    Bool(bool),
    Number(i64),
    Quoted(String),
    RelativeDate(i64, TimeUnit),
    Date(String),
    List(Vec<Value>),
    Function(String, Vec<String>),
    Token(String),
    Reference(ReferenceType, String),
    Subquery(String, Variable, Box<Context>),
}

impl fmt::Display for Value {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Null => write!(f, "null"),
            Bool(b) => write!(f, "{b}"),
            Number(n) => write!(f, "{n}"),
            Quoted(s) => write!(f, "\"{s}\""),
            Reference(r, s) => write!(f, "{}{}", r.symbol(), s),
            RelativeDate(n, unit) => write!(f, "{n}{unit}"),
            Date(s) => write!(f, "\"{s}\""),
            List(arr) => {
                write!(f, "(")?;
                for (i, v) in arr.iter().enumerate() {
                    if i > 0 {
                        write!(f, ", ")?
                    }
                    write!(f, "{v}")?;
                }
                write!(f, ")")
            }
            Function(func, args) => write!(f, "{func}({args:?})"),
            Token(s) => write!(f, "{s}"),
            Subquery(_, variable, _) => write!(f, "(${})", variable.key),
        }
    }
}
