use crate::types::DisplayField;
use serde::{Deserialize, Serialize};
use serde_json::{Value as JsonValue, json};

#[derive(Debug, Serialize, Deserialize)]
pub struct TransformOutput {
    pub data: JsonValue,
    pub error: Option<String>,
    pub success: bool,
    pub fields: Vec<DisplayField>,
}

impl Default for TransformOutput {
    fn default() -> Self {
        Self {
            data: JsonValue::Null,
            error: None,
            success: true,
            fields: vec![],
        }
    }
}

impl TransformOutput {
    pub fn success(data: JsonValue, fields: Vec<DisplayField>) -> Self {
        Self {
            data,
            fields,
            ..Self::default()
        }
    }

    pub fn error(error: &str) -> Self {
        Self {
            error: Some(error.to_string()),
            success: false,
            ..Self::default()
        }
    }

    pub fn serialize(&self) -> String {
        serde_json::to_string(&json!({
            "data": self.data,
            "error": self.error,
            "success": self.success,
            "fields": self.fields.iter().map(|f| {
                json!({
                    "key": f.key(),
                    "name": f.name(),
                    "label": f.label(),
                })
            }).collect::<Vec<_>>(),
        }))
        .unwrap_or_else(|_| "Serialization error".to_string())
    }
}
