use crate::types::Field;
use SortOrder::*;
use serde::{Deserialize, Serialize};
use std::fmt;

#[derive(Debug, PartialEq, Clone, Serialize, Deserialize, Ord, PartialOrd, Eq)]
pub struct Sort {
    pub field: Field,
    pub order: SortOrder,
}

impl From<&str> for Sort {
    fn from(s: &str) -> Self {
        let parts = s.split_whitespace().collect::<Vec<&str>>();

        Sort {
            field: parts[0].into(),
            order: if parts.len() == 1 {
                Ascending
            } else {
                parts[1].into()
            },
        }
    }
}

#[derive(Debug, PartialEq, Clone, Serialize, Deserialize, Ord, PartialOrd, Eq)]
pub enum SortOrder {
    Ascending,
    Descending,
    UnknownSortOrder(String),
}

impl fmt::Display for SortOrder {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Ascending => write!(f, "asc"),
            Descending => write!(f, "desc"),
            UnknownSortOrder(s) => write!(f, "{s}"),
        }
    }
}

impl From<&str> for SortOrder {
    fn from(s: &str) -> Self {
        match s.to_lowercase().as_str() {
            "asc" | "ascending" => Ascending,
            "desc" | "descending" => Descending,
            _ => UnknownSortOrder(s.to_string()),
        }
    }
}
