use crate::types::{Field, Operator, ReferenceType, RelationshipType, RelationshipType::*};
use FieldType::*;
use std::fmt;
use std::ops::BitOr;

#[derive(Debug, Eq, PartialEq, Clone)]
pub enum FieldType {
    Nullable,
    StringLike,
    NumberLike,
    DateLike,
    BooleanLike,
    ListLike(RelationshipType, Box<FieldType>),
    EnumLike(Vec<String>),
    StringEnumLike(Vec<String>),
    ReferenceLike(ReferenceType),

    PairedWith(Box<FieldType>, Vec<Field>),
    WithOperators(Box<FieldType>, Vec<Operator>),
    Multiple(Box<Vec<FieldType>>),
    UnknownFieldType,
}

impl FieldType {
    pub fn is_boolean_like(&self) -> bool {
        self.any(|f| matches!(f, BooleanLike))
    }

    pub fn is_date_like(&self) -> bool {
        self.any(|f| matches!(f, DateLike))
    }

    pub fn is_nullable(&self) -> bool {
        self.any(|f| matches!(f, Nullable))
    }

    pub fn is_list_like(&self) -> bool {
        self.any(|f| matches!(f, ListLike(..)))
    }

    pub fn is_has_many_list_like(&self) -> bool {
        self.any(|f| matches!(f, ListLike(r, _) if *r == HasMany))
    }

    pub fn is_has_one_list_like(&self) -> bool {
        self.any(|f| matches!(f, ListLike(r, _) if *r == HasOne))
    }

    pub fn is_enum_like(&self) -> bool {
        self.any(|f| matches!(f, EnumLike(..) | StringEnumLike(..)))
    }

    pub fn enum_values(&self) -> Vec<String> {
        match self {
            EnumLike(values) | StringEnumLike(values) => values.clone(),
            _ => vec![],
        }
    }

    pub fn find(&self, f: impl Fn(&FieldType) -> bool) -> Option<FieldType> {
        match self {
            WithOperators(t, _) => t.find(f),
            Multiple(v) => v.iter().find(|t| t.any(&f)).cloned(),
            t if f(t) => Some(t.clone()),
            _ => None,
        }
    }

    pub fn any(&self, f: impl Fn(&FieldType) -> bool) -> bool {
        let mut types = vec![self];
        while let Some(field_type) = types.pop() {
            match field_type {
                WithOperators(t, _) => types.push(t),
                Multiple(vec) => types.extend(vec.iter()),
                _ => {
                    if f(field_type) {
                        return true;
                    }
                }
            }
        }
        false
    }

    pub fn with_ops<const N: usize>(&self, operators: [Operator; N]) -> FieldType {
        WithOperators(Box::new(self.to_owned()), operators.to_vec())
    }

    pub fn paired_with<const N: usize>(&self, fields: [Field; N]) -> FieldType {
        PairedWith(Box::new(self.to_owned()), fields.to_vec())
    }
}

impl BitOr for FieldType {
    type Output = Self;

    fn bitor(self, rhs: Self) -> Self {
        match (self, rhs) {
            (Multiple(vec1), Multiple(vec2)) => {
                let mut combined = vec1.to_vec();
                combined.extend(vec2.to_vec());
                Multiple(Box::new(combined))
            }
            (Multiple(vec), other) | (other, Multiple(vec)) => {
                let mut combined = vec.to_vec();
                combined.push(other);
                Multiple(Box::new(combined))
            }
            (lhs, rhs) => Multiple(Box::new(vec![lhs, rhs])),
        }
    }
}

impl fmt::Display for FieldType {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Nullable => write!(f, "`Nullable` (`null`, `none`, `any`)"),
            StringLike => write!(f, "`String`"),
            NumberLike => write!(f, "`Number`"),
            DateLike => write!(f, "`Date`"),
            BooleanLike => write!(f, "`Boolean` (`true`, `false`)"),
            ReferenceLike(r) => write!(f, "{r}"),
            ListLike(..) => write!(f, "`List`"),
            EnumLike(v) => {
                write!(
                    f,
                    "`Enum` ({})",
                    v.iter()
                        .map(|s| format!("`{s}`"))
                        .collect::<Vec<_>>()
                        .join(", ")
                )
            }
            StringEnumLike(v) => {
                write!(
                    f,
                    "`StringEnum` ({})",
                    v.iter()
                        .map(|s| format!("`\"{s}\"`"))
                        .collect::<Vec<_>>()
                        .join(", ")
                )
            }

            // recursive
            WithOperators(field, _) | PairedWith(field, _) => write!(f, "{field}"),
            Multiple(v) => v.iter().try_fold((), |_, field| write!(f, "{field}")),
            UnknownFieldType => write!(f, "`Unsupported`"),
        }
    }
}
