use crate::utils::feature_flags::FeatureFlag;
use Field::*;
use serde::{Deserialize, Serialize};
use std::fmt;

#[derive(Debug, PartialEq, Clone, Serialize, Deserialize, Hash, Eq, PartialOrd, Ord)]
pub enum Field {
    Type,

    Id,
    Label,
    Epic,
    Created,
    Updated,
    Assignee,
    Author,
    Milestone,
    Weight,
    Group,
    Project,
    Confidential,
    Health,
    Closed,
    Iteration,
    Cadence,
    Start,
    Due,
    IncludeSubgroups,
    Title,
    Description,
    Popularity,
    Status,
    MyReaction,
    Subscribed,

    Draft,
    Approver,
    Reviewer,
    Merger,
    State,
    Merged,
    Deployed,
    Environment,
    SourceBranch,
    TargetBranch,

    // Aggregation
    Count,
    TotalTimeToMerge,

    CustomField(String),
    UnknownField(String),
    AliasedField(Box<Field>, String),
}

impl Field {
    pub fn aliased_as(self, alias: &str) -> Field {
        AliasedField(Box::new(self), alias.to_string())
    }

    pub fn dealias(&self) -> &Field {
        match self {
            AliasedField(field, _) => field,
            _ => self,
        }
    }

    pub fn display_field_name(&self) -> String {
        match self.dealias() {
            Assignee => "assignees".to_string(),
            Closed => "closedAt".to_string(),
            Created => "createdAt".to_string(),
            Due => "dueDate".to_string(),
            Health => "healthStatus".to_string(),
            Label => "labels".to_string(),
            Updated => "updatedAt".to_string(),
            Description => "descriptionHtml".to_string(),
            Merged => "mergedAt".to_string(),
            Reviewer => "reviewers".to_string(),
            Merger => "mergeUser".to_string(),
            Approver => "approvedBy".to_string(),
            Start => "startDate".to_string(),
            SourceBranch => "sourceBranch".to_string(),
            TargetBranch => "targetBranch".to_string(),
            TotalTimeToMerge => "totalTimeToMerge".to_string(),

            // Work items feature flag aliases
            Epic if FeatureFlag::GlqlWorkItems.get() => "parent".to_string(),
            Type if FeatureFlag::GlqlWorkItems.get() => "workItemType".to_string(),

            f => f.into(),
        }
    }
}

impl From<String> for Field {
    fn from(s: String) -> Self {
        match s.to_lowercase().as_str() {
            "type" => Type,

            "id" => Id,
            "label" => Label,
            "labels" => Label.aliased_as(s.as_str()),
            "epic" => Epic,
            "created" => Created,
            "opened" | "openedat" | "createdat" => Created.aliased_as(s.as_str()),
            "updated" => Updated,
            "updatedat" => Updated.aliased_as(s.as_str()),
            "assignee" => Assignee,
            "assignees" => Assignee.aliased_as(s.as_str()),
            "author" => Author,
            "milestone" => Milestone,
            "weight" => Weight,
            "group" => Group,
            "project" => Project,
            "confidential" => Confidential,
            "health" => Health,
            "healthstatus" => Health.aliased_as(s.as_str()),
            "closed" => Closed,
            "closedat" => Closed.aliased_as(s.as_str()),
            "iteration" => Iteration,
            "cadence" => Cadence,
            "start" => Start,
            "startdate" => Start.aliased_as(s.as_str()),
            "due" => Due,
            "title" => Title,
            "description" => Description,
            "descriptionhtml" => Description.aliased_as(s.as_str()),
            "popularity" => Popularity,
            "duedate" => Due.aliased_as(s.as_str()),
            "includesubgroups" => IncludeSubgroups.aliased_as(s.as_str()),
            "status" => Status,
            "myreaction" => MyReaction.aliased_as(s.as_str()),
            "myreactionemoji" => MyReaction.aliased_as(s.as_str()),
            "subscribed" => Subscribed,

            "reviewer" => Reviewer,
            "reviewers" | "reviewedby" => Reviewer.aliased_as(s.as_str()),
            "merger" => Merger,
            "mergedby" | "mergeuser" => Merger.aliased_as(s.as_str()),
            "draft" => Draft,
            "approver" => Approver,
            "approvers" | "approvedby" => Approver.aliased_as(s.as_str()),
            "state" => State,
            "merged" => Merged,
            "mergedat" => Merged.aliased_as(s.as_str()),
            "deployed" => Deployed,
            "deployedat" => Deployed.aliased_as(s.as_str()),
            "environment" => Environment,
            "sourcebranch" | "sourcebranches" => SourceBranch.aliased_as(s.as_str()),
            "targetbranch" | "targetbranches" => TargetBranch.aliased_as(s.as_str()),

            "count" => Count,
            "totaltimetomerge" => TotalTimeToMerge.aliased_as(s.as_str()),

            _ => UnknownField(s),
        }
    }
}

impl From<&Field> for String {
    fn from(field: &Field) -> Self {
        match field {
            UnknownField(s) | AliasedField(_, s) => s.clone(),
            CustomField(s) => format!("customField(\"{s}\")"),
            _ => format!("{field:?}").to_lowercase(),
        }
    }
}

impl fmt::Display for Field {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "{}", String::from(self))
    }
}

impl From<&str> for Field {
    fn from(s: &str) -> Self {
        Field::from(s.to_string())
    }
}
