use crate::{
    types::Field::{self, *},
    utils::common::to_sentence_case,
};
use DisplayField::*;
use serde::{Deserialize, Serialize};

#[derive(Debug, PartialEq, Clone, Serialize, Deserialize, Hash, Eq, PartialOrd, Ord)]
pub enum DisplayField {
    Static(Field),
    FieldFunction(String, Vec<String>),
    AliasedDisplayField(Box<DisplayField>, String),
}

impl DisplayField {
    // Used for indexing the data in the API response
    pub fn key(&self) -> String {
        match self {
            Static(field) => match field {
                AliasedField(_, alias) => alias.clone(),
                _ => field.to_string(),
            },
            FieldFunction(func, args) => format!("{}_{}", func, args.join("_")),
            AliasedDisplayField(field, _) => field.key(),
        }
    }

    // Used for fetching data from the API
    pub fn name(&self) -> String {
        match self {
            Static(field) => field.display_field_name(),
            FieldFunction(func, _) => match func.as_str() {
                "labels" => "labels".to_string(),
                _ => func.clone(),
            },
            AliasedDisplayField(field, _) => field.dealias().name(),
        }
    }

    // Used for displaying the field in the UI
    pub fn label(&self) -> String {
        match self {
            Static(_) => to_sentence_case(&self.key()),
            FieldFunction(func, args) => {
                // format: Func: Arg1, Arg2, ...
                format!(
                    "{}: {}",
                    to_sentence_case(func),
                    args.iter()
                        .map(|a| to_sentence_case(a))
                        .collect::<Vec<_>>()
                        .join(", ")
                )
            }
            AliasedDisplayField(_, alias) => alias.clone(),
        }
    }
}

impl From<Field> for DisplayField {
    fn from(field: Field) -> Self {
        Static(field)
    }
}

impl From<&str> for DisplayField {
    fn from(s: &str) -> Self {
        Static(s.into())
    }
}

impl DisplayField {
    pub fn aliased_as(self, alias: &str) -> DisplayField {
        AliasedDisplayField(Box::new(self), alias.to_string())
    }

    pub fn dealias(&self) -> &DisplayField {
        match self {
            AliasedDisplayField(field, _) => field,
            _ => self,
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_display_field() {
        let field = Static(Field::from("id"));
        assert_eq!(field.label(), "ID");
        assert_eq!(field.name(), "id");
        assert_eq!(field.key(), "id");

        let field = Static(Field::from("labels"));
        assert_eq!(field.label(), "Labels");
    }
}
