use crate::types::{
    DisplayField, Field::*, Sort, Source, Source::*, Variable, aggregation::AggregationContext,
};
use chrono::{DateTime, Utc};
use serde::{Deserialize, Serialize};

#[derive(Debug, PartialEq, Clone, Serialize, Deserialize, Ord, PartialOrd, Eq)]
pub struct Context {
    pub time: DateTime<Utc>,
    pub source: Option<Source>,
    pub username: Option<String>,
    pub fields: Vec<DisplayField>,
    pub project: Option<String>,
    pub group: Option<String>,
    pub sort: Option<Sort>,
    pub variables: Vec<Variable>,
    pub is_subquery: bool,
    pub aggregate: Option<AggregationContext>,
}

impl Default for Context {
    fn default() -> Self {
        Self {
            time: Utc::now(),
            username: None,
            fields: vec![Id.into(), Title.into(), Description.into()],
            project: None,
            group: None,
            source: Some(Issues),
            sort: None,
            aggregate: None,
            variables: vec![],
            is_subquery: false,
        }
    }
}

impl Context {
    pub fn with_project(project: &str) -> Self {
        Context {
            project: Some(project.to_string()),
            ..Context::default()
        }
    }

    pub fn with_group(group: &str) -> Self {
        Context {
            group: Some(group.to_string()),
            ..Context::default()
        }
    }

    pub fn include_page_info(&self) -> bool {
        !self.is_subquery
    }
}
