use crate::types::{DisplayField, Variable};
use serde::{Deserialize, Serialize};
use serde_json::{Map, json};

#[derive(Debug, Serialize, Deserialize)]
pub struct CompileOutput {
    pub output: String,
    pub success: bool,
    pub variables: Vec<Variable>,
    pub fields: Vec<DisplayField>,
}

impl Default for CompileOutput {
    fn default() -> Self {
        Self {
            output: "".to_string(),
            success: true,
            variables: vec![],
            fields: vec![],
        }
    }
}

impl CompileOutput {
    pub fn success(output: String) -> Self {
        Self {
            output,
            ..Self::default()
        }
    }

    pub fn error(error: &str) -> Self {
        Self {
            output: error.to_string(),
            success: false,
            ..Self::default()
        }
    }

    pub fn with_variables(self, variables: Vec<Variable>) -> Self {
        Self { variables, ..self }
    }

    pub fn with_fields(self, fields: Vec<DisplayField>) -> Self {
        Self { fields, ..self }
    }

    pub fn serialize(&self) -> String {
        serde_json::to_string(&json!({
            "success": self.success,
            "output": self.output,
            "variables": self.variables.iter().fold(Map::new(), |mut acc, v| {
                acc.insert(v.key.clone(), json!({
                    "type": v.r#type,
                    "value": v.value,
                }));
                acc
            }),
            "fields": self.fields.iter().map(|f| {
                json!({
                    "key": f.key(),
                    "name": f.name(),
                    "label": f.label(),
                })
            }).collect::<Vec<_>>(),
        }))
        .unwrap_or_else(|_| "Serialization error".to_string())
    }
}
