use crate::{
    errors::{GlqlError, GlqlErrorKind::*},
    types::{
        Context, Query,
        ReferenceType::*,
        Value::{self, *},
    },
    utils::date_time::{calculate_days, start_of_day},
};
use chrono::NaiveDate;

pub fn apply_functions(query: &mut Query, ctx: &Context) -> Result<(), GlqlError> {
    for expr in query.expressions.iter_mut() {
        expr.value = process_value(&expr.value, ctx)?;
    }
    Ok(())
}

fn process_value(value: &Value, ctx: &Context) -> Result<Value, GlqlError> {
    match value {
        Function(func_name, func_arg) => evaluate_function(func_name, func_arg, ctx),
        Date(date_str) => Ok(Quoted(validate_date_format(date_str)?)),
        RelativeDate(quantity, unit) => {
            let days = calculate_days(*quantity, unit.clone(), ctx.time).map_err(|e| {
                GlqlError::analyzer_error(InvalidDateFormat {
                    date: e.to_string(),
                })
            })?;
            Ok(Quoted(start_of_day(days, ctx.time)))
        }
        List(arr) => {
            let processed_arr: Result<Vec<Value>, GlqlError> =
                arr.iter().map(|v| process_value(v, ctx)).collect();
            Ok(List(processed_arr?))
        }
        _ => Ok(value.clone()),
    }
}

fn validate_date_format(date_str: &str) -> Result<String, GlqlError> {
    // Try to parse the date string
    match NaiveDate::parse_from_str(date_str, "%Y-%m-%d") {
        Ok(_) => Ok(date_str.to_string()),
        Err(_) => Err(GlqlError::analyzer_error(InvalidDateFormat {
            date: date_str.into(),
        })),
    }
}

fn evaluate_function(
    func_name: &str,
    func_arg: &[String],
    ctx: &Context,
) -> Result<Value, GlqlError> {
    match func_name.to_lowercase().as_str() {
        "today" => Ok(Quoted(start_of_day(0, ctx.time))),
        "startofday" => {
            let day = func_arg[0]
                .parse::<i64>()
                .map_err(|_| GlqlError::analyzer_error(UnknownError))?;

            Ok(Quoted(start_of_day(day, ctx.time)))
        }

        "currentuser" => {
            if let Some(user) = &ctx.username {
                Ok(Reference(UserRef, user.clone()))
            } else {
                Ok(Null)
            }
        }
        _ => Err(GlqlError::analyzer_error(UnrecognizedFunction {
            name: func_name.into(),
            args: func_arg.to_vec(),
        })),
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::types::{Expression, Field::*, Operator::*};
    use crate::utils::date_time::TimeUnit::*;
    use chrono::{DateTime, TimeZone, Utc};

    fn create_test_context(current_time: DateTime<Utc>) -> Context {
        Context {
            username: Some("testuser".to_string()),
            time: current_time,
            ..Context::default()
        }
    }

    #[allow(clippy::useless_vec)]
    #[test]
    fn test_evaluate_function() {
        macro_rules! valid_test_cases {
            ($($name:ident: ($func_name:expr, $input:expr) => $expected:expr)*) => {
                $(
                    fn $name() {
                        let ctx = create_test_context(Utc.with_ymd_and_hms(2023, 5, 15, 12, 0, 0).unwrap());
                        let result = evaluate_function($func_name, &vec![$input.to_string()], &ctx).unwrap();
                        assert_eq!(result, $expected);
                    }

                    $name();
                )*
            };
        }

        valid_test_cases! {
            test_today: ("today", "") => Quoted("2023-05-15".to_string())
            test_current_user: ("currentUser", "") => Reference(UserRef, "testuser".to_string())
            test_start_of_day_negative: ("startOfDay", "-1") => Quoted("2023-05-14".to_string())
            test_start_of_day_positive: ("startOfDay", "5") => Quoted("2023-05-20".to_string())
            test_today_case_insensitive: ("TODAY", "") => Quoted("2023-05-15".to_string())
            test_current_user_case_insensitive: ("CURRENTUser", "") => Reference(UserRef, "testuser".to_string())
            test_start_of_day_case_insensitive: ("StaRtOfDAy", "10") => Quoted("2023-05-25".to_string())
        }

        let ctx = create_test_context(Utc.with_ymd_and_hms(2023, 5, 15, 12, 0, 0).unwrap());
        match evaluate_function("invalidFunction", &[], &ctx) {
            Err(e) => assert_eq!(
                e.to_string(),
                "Error: Unrecognized function: invalidFunction()"
            ),
            _ => unreachable!("Expected error for invalid function"),
        }
    }

    #[test]
    fn test_apply_functions() {
        let ctx = create_test_context(Utc.with_ymd_and_hms(2023, 5, 15, 12, 0, 0).unwrap());
        let mut query = Query {
            expressions: vec![
                Expression::new(Due, Equal, Function("today".into(), vec![])),
                Expression::new(
                    Assignee,
                    In,
                    List(vec![
                        Quoted("foo".to_string()),
                        Function("currentUser".to_string(), vec![]),
                    ]),
                ),
                Expression::new(Closed, LessThan, RelativeDate(-1, Days)),
                Expression::new(Updated, GreaterThan, Quoted("2024-01-01".to_string())),
            ],
        };

        macro_rules! assert_expressions {
            ($value:expr, $expected:expr) => {
                assert_eq!($value, &Quoted($expected.to_string()))
            };
        }

        let result = apply_functions(&mut query, &ctx);
        assert!(result.is_ok());

        // Test: due = today()
        assert_expressions!(&query.expressions[0].value, "2023-05-15");

        // Test: assignee in ("foo", currentUser())
        match &query.expressions[1].value {
            List(arr) => {
                assert_eq!(arr.len(), 2);
                assert_expressions!(&arr[0], "foo");
                assert_eq!(&arr[1], &Reference(UserRef, "testuser".to_string()));
            }
            _ => unreachable!("Expected GlqlArray for assignee"),
        }

        // Test: closed < -1d
        assert_expressions!(&query.expressions[2].value, "2023-05-14");

        // Test: updated > 2024-01-01
        assert_expressions!(&query.expressions[3].value, "2024-01-01");
    }
}
