use crate::errors::GlqlError;
use crate::types::DisplayField::{self, *};
use crate::utils::common::wildcard_match;
use serde_json::{Value as JsonValue, json};

pub fn transform_fields(
    data: JsonValue,
    fields: &Vec<DisplayField>,
) -> Result<JsonValue, GlqlError> {
    let mut data = data.clone();

    for display_field in fields {
        let field_key = display_field.key();
        let field_name = display_field.name();

        for i in 0.. {
            if data["nodes"][i].is_null() {
                break;
            }

            match display_field.dealias() {
                FieldFunction(name, args) => match name.as_str() {
                    "labels" => {
                        let mut filtered: Vec<JsonValue> = vec![];
                        let mut remaining: Vec<JsonValue> = vec![];

                        for j in 0.. {
                            let label = &data["nodes"][i]["labels"]["nodes"][j];
                            if label.is_null() {
                                break;
                            }

                            let label_title = label["title"].as_str().unwrap_or("");
                            if args.iter().any(|value| wildcard_match(label_title, value)) {
                                filtered.push(label.clone())
                            } else {
                                remaining.push(label.clone())
                            }
                        }

                        data["nodes"][i][&field_key] = json!({ "nodes": filtered });
                        data["nodes"][i]["labels"] = json!({ "nodes": remaining });
                    }
                    _ => break,
                },

                _ => {
                    data["nodes"][i][&field_key] = match field_key.as_str() {
                        "lastComment" => {
                            data["nodes"][i]["lastComment"]["nodes"][0]["bodyHtml"].clone()
                        }
                        _ => data["nodes"][i][&field_name].clone(),
                    };
                }
            }
        }
    }

    Ok(data)
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::types::DisplayField;
    use crate::types::Field::{self, *};
    use serde_json::json;

    #[test]
    fn test_transform_fields() {
        let fields = vec![
            DisplayField::Static(Field::Id),
            DisplayField::FieldFunction(
                "labels".to_string(),
                vec!["bug".to_string(), "critical".to_string()],
            ),
            DisplayField::FieldFunction(
                "labels".to_string(),
                vec!["feature*".to_string(), "*maintenance*".to_string()],
            )
            .aliased_as("Feature/Maintenance"),
        ];

        let mock_issues = json!({
            "nodes": [
                {
                    "id": "1",
                    "title": "Issue 1",
                    "labels": {
                        "nodes": [
                            { "title": "bug::closed" },
                            { "title": "bug" },
                            { "title": "critical" },
                            { "title": "feature-request" }
                        ]
                    }
                },
                {
                    "id": "2",
                    "title": "Issue 2",
                    "labels": {
                        "nodes": [
                            { "title": "feature-request" },
                            { "title": "enhancement" },
                            { "title": "a-maintenance-b" },
                            { "title": "documentation" }
                        ]
                    }
                }
            ],
            "count": 100
        });

        let result = transform_fields(mock_issues.clone(), &fields);
        let expected = json!({
            "nodes": [
                {
                    "id": "1",
                    "title": "Issue 1",
                    "labels_bug_critical": {
                        "nodes": [
                            { "title": "bug" },
                            { "title": "critical" }
                        ]
                    },
                    "labels_feature*_*maintenance*": {
                        "nodes": [
                            { "title": "feature-request" }
                        ]
                    },
                    "labels": {
                        "nodes": [
                            { "title": "bug::closed" }
                        ]
                    }
                },
                {
                    "id": "2",
                    "title": "Issue 2",
                    "labels_bug_critical": {
                        "nodes": []
                    },
                    "labels_feature*_*maintenance*": {
                        "nodes": [
                            { "title": "feature-request" },
                            { "title": "a-maintenance-b" }
                        ]
                    },
                    "labels": {
                        "nodes": [
                            { "title": "enhancement" },
                            { "title": "documentation" }
                        ]
                    }
                }
            ],
            "count": 100
        });

        assert_eq!(result.unwrap(), expected);
    }

    #[test]
    fn test_derived_fields_and_aliases() {
        let mock_data = json!({
            "nodes": [
                {
                    "id": "1",
                    "title": "Issue with login",
                    "createdAt": "2024-01-15T10:30:00Z",
                    "lastComment": {
                        "nodes": [
                            { "bodyHtml": "This is the last comment" }
                        ]
                    }
                },
                {
                    "id": "2",
                    "title": "Feature request",
                    "createdAt": "2024-01-16T14:20:00Z",
                    "lastComment": {
                        "nodes": []
                    }
                }
            ],
            "count": 100
        });

        // Test fields with aliases: createdAt -> created, lastComment -> lastComment
        let fields = vec![
            Static(Id),
            Static(Title),
            Static("lastComment".into()),
            Static(Created),
        ];

        let result = transform_fields(mock_data.clone(), &fields);
        let expected = json!({
            "nodes": [
                {
                    "id": "1",
                    "title": "Issue with login",
                    "createdAt": "2024-01-15T10:30:00Z",
                    "created": "2024-01-15T10:30:00Z",
                    "lastComment": "This is the last comment" ,
                },
                {
                    "id": "2",
                    "title": "Feature request",
                    "createdAt": "2024-01-16T14:20:00Z",
                    "created": "2024-01-16T14:20:00Z",
                    "lastComment": JsonValue::Null,
                }
            ],
            "count": 100
        });

        assert_eq!(result.unwrap(), expected);
    }
}
