use crate::errors::{GlqlError, GlqlErrorKind::*};
use crate::types::{Expression, Field, Operator, Operator::*, Source, Value, Value::*};
use chrono::NaiveDate;
use indexmap::map::IndexMap;

pub fn combine_expressions(expressions: Vec<Expression>) -> Vec<Expression> {
    let mut grouped: IndexMap<(Field, Operator), Vec<Value>> = IndexMap::new();

    for expr in expressions {
        grouped
            .entry((expr.field.clone(), expr.operator.clone()))
            .or_default()
            .push(expr.value);
    }

    grouped
        .into_iter()
        .map(|((field, operator), values)| {
            Expression::new(
                field,
                operator,
                values
                    .into_iter()
                    .reduce(|acc, val| combine_values(&acc, &val).unwrap_or(acc))
                    .unwrap(),
            )
        })
        .collect()
}

fn combine_values(a: &Value, b: &Value) -> Result<Value, GlqlError> {
    match (a, b) {
        (Quoted(s1) | Reference(_, s1), Quoted(s2) | Reference(_, s2)) if s1 == s2 => Ok(a.clone()),
        (Quoted(_) | Reference(..), Quoted(_) | Reference(..)) => {
            Ok(List(vec![a.clone(), b.clone()]))
        }
        (List(arr1), List(arr2)) if arr1 == arr2 => Ok(a.clone()),
        (List(arr1), List(arr2)) => {
            let mut combined: Vec<Value> = arr1.iter().chain(arr2.iter()).cloned().collect();
            combined.sort();
            combined.dedup();
            Ok(List(combined))
        }
        (List(arr), Quoted(s) | Reference(_, s)) | (Quoted(s) | Reference(_, s), List(arr)) => {
            let mut new_arr = arr.clone();
            if !new_arr.contains(&Quoted(s.clone())) {
                new_arr.push(Quoted(s.clone()));
            }
            Ok(List(new_arr))
        }
        _ => Err(GlqlError::analyzer_error(CannotCombineValues)),
    }
}

pub fn expand_expression(expr: &Expression, source: &Source) -> Vec<Expression> {
    if source.analyzer().field_type(&expr.field).is_date_like()
        && expr.operator == Equal
        && is_date_castable(&expr.value)
    {
        vec![
            Expression::new(expr.field.clone(), GreaterThanEquals, expr.value.clone()),
            Expression::new(expr.field.clone(), LessThanEquals, expr.value.clone()),
        ]
    } else {
        vec![expr.clone()]
    }
}

fn is_falsey(v: &Value) -> bool {
    match v {
        Null | Bool(false) => true,
        Number(n) if *n == 0 => true,
        Quoted(s) => s.to_lowercase() == "false",
        _ => false,
    }
}

fn transform_date(v: &Value, o: &Operator) -> Value {
    match (v, o) {
        (Date(s) | Quoted(s), LessThan | GreaterThanEquals) => Date(format!("{s} 00:00")),
        (Date(s) | Quoted(s), GreaterThan | LessThanEquals) => Date(format!("{s} 23:59")),
        _ => v.clone(),
    }
}

fn is_truthy(v: &Value) -> bool {
    !is_falsey(v)
}

fn truthy(x: &Value) -> Value {
    if is_truthy(x) {
        Bool(true)
    } else {
        Bool(false)
    }
}

fn is_date_castable(v: &Value) -> bool {
    match v {
        Date(_) => true,
        RelativeDate(_, _) => true,
        Quoted(s) => NaiveDate::parse_from_str(s, "%Y-%m-%d").is_ok(),
        _ => false,
    }
}

pub fn transform_expression(expr: &Expression, source: &Source) -> Expression {
    let mut f = expr.field.dealias().clone();
    let mut o = expr.operator.clone();
    let mut v = expr.value.clone();

    let field_type = source.analyzer().field_type(&f);
    let token = |v: &str| Token(v.into());

    (f, o, v) = match (&f, &o, &v) {
        (_, NotEqual, List(arr)) if arr.is_empty() => (f, Equal, token("ANY")),
        (_, Equal, List(arr)) if arr.is_empty() => (f, o, token("NONE")),
        (_, NotEqual, Bool(b)) => (f, Equal, Bool(!b)),
        (_, GreaterThan | LessThan | GreaterThanEquals | LessThanEquals, _) => {
            (f, o.clone(), transform_date(&v, &o))
        }

        (.., Bool(_)) if field_type.is_boolean_like() => (f, o, v),
        (..) if field_type.is_boolean_like() => (f, o, truthy(&v)),

        (_, In, List(_)) if field_type.is_has_one_list_like() => (f, Equal, v),
        (_, NotEqual, Token(w)) if field_type.is_nullable() && w == "ANY" => {
            (f, Equal, token("NONE"))
        }
        (_, NotEqual, Token(w)) if field_type.is_nullable() && w == "NONE" => {
            (f, Equal, token("ANY"))
        }
        _ => (f, o, v),
    };

    Expression::new(f, o, v)
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::types::{Field::*, Source::*};

    #[test]
    fn test_combine_values() {
        // Test combining two equal strings
        assert_eq!(
            combine_values(&Quoted("foo".to_string()), &Quoted("foo".to_string())),
            Ok(Quoted("foo".to_string()))
        );

        // Test combining two different strings
        assert_eq!(
            combine_values(&Quoted("foo".to_string()), &Quoted("bar".to_string())),
            Ok(List(vec![
                Quoted("foo".to_string()),
                Quoted("bar".to_string())
            ]))
        );

        // Test combining two equal arrays
        let arr = vec![Quoted("foo".to_string())];
        assert_eq!(
            combine_values(&List(arr.clone()), &List(arr.clone())),
            Ok(List(arr))
        );

        // Test combining two different arrays
        assert_eq!(
            combine_values(
                &List(vec![Quoted("foo".to_string())]),
                &List(vec![Quoted("bar".to_string()), Quoted("baz".to_string())])
            ),
            Ok(List(vec![
                Quoted("bar".to_string()),
                Quoted("baz".to_string()),
                Quoted("foo".to_string())
            ]))
        );

        // Test combining array and string
        assert_eq!(
            combine_values(
                &List(vec![Quoted("foo".to_string())]),
                &Quoted("bar".to_string())
            ),
            Ok(List(vec![
                Quoted("foo".to_string()),
                Quoted("bar".to_string())
            ]))
        );

        // Test combining incompatible types
        assert_eq!(
            combine_values(&Number(1), &Quoted("foo".to_string())),
            Err(GlqlError::analyzer_error(CannotCombineValues))
        );
    }

    #[test]
    fn test_combine_expressions() {
        let expressions = vec![
            Expression::new(Label, Equal, Quoted("bug".into())),
            Expression::new(Label, Equal, Quoted("feature".into())),
            Expression::new(Author, Equal, Quoted("foo".into())),
        ];

        let combined = combine_expressions(expressions);

        assert_eq!(combined.len(), 2);

        // Check the combined label expression
        let label_expr = combined.iter().find(|e| e.field == Label).unwrap();
        assert_eq!(label_expr.operator, Equal);
        assert_eq!(
            label_expr.value,
            List(vec![
                Quoted("bug".to_string()),
                Quoted("feature".to_string())
            ])
        );

        // Check the author expression (should remain unchanged)
        let author_expr = combined.iter().find(|e| e.field == Author).unwrap();
        assert_eq!(author_expr.operator, Equal);
        assert_eq!(author_expr.value, Quoted("foo".to_string()));
    }

    #[test]
    fn test_transform_expression_with_in_operator() {
        let test_cases = vec![
            (
                "milestone",
                vec![
                    Quoted("Milestone 1".to_string()),
                    Quoted("Milestone 2".to_string()),
                ],
            ),
            (
                "iteration",
                vec![
                    Quoted("Iteration 1".to_string()),
                    Quoted("Iteration 2".to_string()),
                ],
            ),
            (
                "cadence",
                vec![
                    Quoted("Cadence 1".to_string()),
                    Quoted("Cadence 2".to_string()),
                ],
            ),
        ];

        for (field, values) in test_cases {
            let expr = Expression::new(field.into(), In, List(values.clone()));
            let transformed_expr = transform_expression(&expr, &Issues);

            assert_eq!(transformed_expr.field, field.into());
            assert_eq!(transformed_expr.operator, Equal);
            assert_eq!(transformed_expr.value, List(values));
        }
    }
}
