use crate::{
    errors::{GlqlError, GlqlErrorKind, InvalidScopeType},
    transformer::field_functions::transform_fields,
    types::DisplayField,
    utils::{date_time::TimeRange, feature_flags::FeatureFlag},
};
use serde_json::{Map, Value as JsonValue, json};

/// Data sources that can be transformed
const DATA_SOURCES: [&str; 4] = ["epics", "issues", "mergeRequests", "workItems"];

/// Transforms work items by merging widget data into the work item
/// and removing the widgets field
pub fn transform_work_items(data: &JsonValue) -> Result<JsonValue, GlqlError> {
    let mut result = data.clone();
    for i in 0.. {
        let work_item = &data["nodes"][i];
        if work_item.is_null() {
            break;
        }

        for j in 0.. {
            let widget = &work_item["widgets"][j];
            if widget.is_null() {
                break;
            }

            for (key, value) in widget.as_object().unwrap_or(&Map::new()) {
                if key != "type" && key != "__typename" {
                    result["nodes"][i][key] = value.clone();
                }
            }
        }

        result["nodes"][i]["widgets"].take();
    }

    Ok(result)
}

fn transform_aggregated_data(
    data: &JsonValue,
    dimension_name: &str,
) -> Result<JsonValue, GlqlError> {
    let mut result = vec![];
    if let Some(data) = data.as_object() {
        for (dimension_key, metrics_obj) in data {
            if dimension_key == "__typename" {
                continue;
            }

            if !dimension_key.starts_with("from_") {
                // skip the transformation if the data is not aggregated
                return Ok(json!(data));
            }

            let mut item = json!({
                "__typename": "GlqlAggregatedData",
                dimension_name: {
                    "__typename": "GlqlDimension",
                    "title": match TimeRange::from_key(dimension_key) {
                        Ok(time_range) => time_range.stringify(),
                        Err(_) => dimension_key.to_string(),
                    },
                },
            });

            if let Some(metrics) = metrics_obj.as_object() {
                for (metric_key, metric_value) in metrics {
                    item[metric_key] = metric_value.clone();
                }
            }

            result.push(item);
        }
    }

    Ok(json!({
        "__typename": "GlqlAggregatedDataConnection",
        "nodes": result,
        "count": result.len(),
    }))
}

pub fn transform_for_data_source(data: &JsonValue) -> Result<JsonValue, GlqlError> {
    let map = Map::new();
    let scope = data["project"]
        .as_object()
        .unwrap_or(data["group"].as_object().unwrap_or(&map));

    if scope.is_empty() {
        return Err(GlqlError::analyzer_error(GlqlErrorKind::InvalidScope {
            scope_type: if data.get("project").is_some() && data["project"].is_null() {
                InvalidScopeType::Project
            } else if data.get("group").is_some() && data["group"].is_null() {
                InvalidScopeType::Group
            } else {
                InvalidScopeType::Namespace
            },
        }));
    }

    for source in DATA_SOURCES {
        if let Some(data) = scope.get(source) {
            let mut transformed = data.clone();

            if source == "workItems" {
                FeatureFlag::GlqlWorkItems.set(true);
                transformed = transform_work_items(&transformed)?;
            } else {
                FeatureFlag::GlqlWorkItems.set(false);
            }

            return Ok(transformed);
        }
    }

    Ok(json!(scope))
}

pub fn transform(data: &JsonValue, fields: &Vec<DisplayField>) -> Result<JsonValue, GlqlError> {
    let data = transform_for_data_source(data)?;
    let data = transform_aggregated_data(&data, fields[0].name().as_str())?;

    transform_fields(data, fields)
}

#[cfg(test)]
mod tests {
    use super::*;
    use serde_json::json;

    // Mock data functions matching the JavaScript tests
    fn mock_labels1() -> JsonValue {
        json!({ "nodes": [{ "title": "bug" }] })
    }

    fn mock_labels2() -> JsonValue {
        json!({ "nodes": [{ "title": "feature" }] })
    }

    fn mock_issues() -> JsonValue {
        json!({
            "issues": {
                "nodes": [
                    { "id": "1", "title": "Lorem ipsum", "labels": mock_labels1() },
                    { "id": "2", "title": "Dolor sit amet", "labels": mock_labels2() },
                ]
            }
        })
    }

    fn mock_epics() -> JsonValue {
        json!({
            "epics": {
                "nodes": [
                    { "id": "1", "title": "Lorem ipsum", "labels": mock_labels1() },
                    { "id": "2", "title": "Dolor sit amet", "labels": mock_labels2() },
                ]
            }
        })
    }

    fn mock_merge_requests() -> JsonValue {
        json!({
            "mergeRequests": {
                "nodes": [
                    { "id": "1", "title": "Lorem ipsum", "labels": mock_labels1() },
                    { "id": "2", "title": "Dolor sit amet", "labels": mock_labels2() },
                ]
            }
        })
    }

    fn mock_work_items() -> JsonValue {
        json!({
            "workItems": {
                "nodes": [
                    {
                        "id": "1",
                        "title": "Lorem ipsum",
                        "widgets": [
                            {},
                            {},
                            {},
                            { "__typename": "WorkItemWidgetLabels", "type": "LABELS", "labels": mock_labels1() },
                        ],
                    },
                    {
                        "id": "2",
                        "title": "Dolor sit amet",
                        "widgets": [
                            {},
                            {},
                            {},
                            { "__typename": "WorkItemWidgetLabels", "type": "LABELS", "labels": mock_labels2() },
                        ],
                    },
                ]
            }
        })
    }

    fn mock_work_items_without_widgets() -> JsonValue {
        json!({
            "workItems": {
                "nodes": [
                    { "id": "1", "title": "Lorem ipsum" },
                    { "id": "2", "title": "Dolor sit amet" },
                ]
            }
        })
    }

    #[test]
    fn test_transform_for_data_source_issues() {
        let mock_data = json!({ "project": mock_issues() });
        let result = transform_for_data_source(&mock_data);

        assert!(result.is_ok());
        assert_eq!(result.unwrap(), mock_issues()["issues"]);
    }

    #[test]
    fn test_transform_for_data_source_epics() {
        let mock_data = json!({ "project": mock_epics() });
        let result = transform_for_data_source(&mock_data);

        assert!(result.is_ok());
        assert_eq!(result.unwrap(), mock_epics()["epics"]);
    }

    #[test]
    fn test_transform_for_data_source_merge_requests() {
        let mock_data = json!({ "project": mock_merge_requests() });
        let result = transform_for_data_source(&mock_data);

        assert!(result.is_ok());
        assert_eq!(result.unwrap(), mock_merge_requests()["mergeRequests"]);
    }

    #[test]
    fn test_transform_for_data_source_work_items() {
        let mock_data = json!({ "project": mock_work_items() });
        let result = transform_for_data_source(&mock_data);

        assert!(result.is_ok());

        // Work items should be transformed (widgets merged and removed)
        let expected = json!({
            "nodes": [
                {
                    "id": "1",
                    "title": "Lorem ipsum",
                    "labels": mock_labels1(),
                    "widgets": JsonValue::Null
                },
                {
                    "id": "2",
                    "title": "Dolor sit amet",
                    "labels": mock_labels2(),
                    "widgets": JsonValue::Null
                },
            ]
        });
        assert_eq!(result.unwrap(), expected);
    }

    #[test]
    fn test_transform_for_data_source_group_scope() {
        let mock_data = json!({ "group": JsonValue::Null });
        let result = transform_for_data_source(&mock_data);

        assert!(result.is_err());
        assert_eq!(
            result.err().unwrap().to_string(),
            "Error: Group does not exist or you do not have access to it"
        );

        let mock_data = json!({ "project": JsonValue::Null });
        let result = transform_for_data_source(&mock_data);

        assert!(result.is_err());
        assert_eq!(
            result.err().unwrap().to_string(),
            "Error: Project does not exist or you do not have access to it"
        );
    }

    #[test]
    fn test_transform_for_data_source_no_matching_source() {
        let mock_data = json!({
            "project": {
                "unknownSource": { "nodes": [] }
            }
        });
        let result = transform_for_data_source(&mock_data);
        assert!(result.is_ok());
        assert_eq!(result.unwrap(), mock_data["project"]);
    }

    #[test]
    fn test_transform_for_data_source_no_scope() {
        let mock_data = json!({ "other": mock_issues() });
        let result = transform_for_data_source(&mock_data);
        assert!(result.is_err());
        assert_eq!(
            result.err().unwrap().to_string(),
            "Error: Namespace does not exist or you do not have access to it"
        );
    }

    #[test]
    fn test_transform_work_items_with_widgets() {
        let work_items = mock_work_items()["workItems"].clone();
        let result = transform_work_items(&work_items);

        let expected = json!({
            "nodes": [
                {
                    "id": "1",
                    "title": "Lorem ipsum",
                    "labels": mock_labels1(),
                    "widgets": JsonValue::Null
                },
                {
                    "id": "2",
                    "title": "Dolor sit amet",
                    "labels": mock_labels2(),
                    "widgets": JsonValue::Null
                },
            ]
        });
        assert_eq!(result.unwrap(), expected);
    }

    #[test]
    fn test_transform_work_items_without_widgets() {
        let work_items = mock_work_items_without_widgets()["workItems"].clone();
        let result = transform_work_items(&work_items);

        // Should remain unchanged since there are no widgets
        let expected = json!({
            "nodes": [
                { "id": "1", "title": "Lorem ipsum", "widgets": JsonValue::Null },
                { "id": "2", "title": "Dolor sit amet", "widgets": JsonValue::Null },
            ]
        });
        assert_eq!(result.unwrap(), expected);
    }

    #[test]
    fn test_transform_work_items_empty_widgets() {
        let work_items = json!({
            "workItems": {
                "nodes": [
                    {
                        "id": "1",
                        "title": "Lorem ipsum",
                        "widgets": [],
                    },
                ]
            }
        });
        let work_items_data = work_items["workItems"].clone();
        let result = transform_work_items(&work_items_data);

        let expected = json!({
            "nodes": [
                {
                    "id": "1",
                    "title": "Lorem ipsum",
                    "widgets": JsonValue::Null,
                },
            ]
        });
        assert_eq!(result.unwrap(), expected);
    }

    #[test]
    fn test_transform_work_items_complex_widgets() {
        let work_items = json!({
            "workItems": {
                "nodes": [
                    {
                        "id": "1",
                        "title": "Lorem ipsum",
                        "widgets": [
                            { "__typename": "WorkItemWidgetLabels", "type": "LABELS", "labels": mock_labels1() },
                            { "__typename": "WorkItemWidgetDescription", "type": "DESCRIPTION", "description": "Test description" },
                            { "__typename": "WorkItemWidgetAssignees", "type": "ASSIGNEES", "assignees": { "nodes": [{ "name": "John" }] } },
                        ],
                    },
                ]
            }
        });
        let work_items_data = work_items["workItems"].clone();
        let result = transform_work_items(&work_items_data);

        let expected = json!({
            "nodes": [
                {
                    "id": "1",
                    "title": "Lorem ipsum",
                    "labels": mock_labels1(),
                    "description": "Test description",
                    "assignees": { "nodes": [{ "name": "John" }] },
                    "widgets": JsonValue::Null,
                },
            ]
        });
        assert_eq!(result.unwrap(), expected);
    }

    #[test]
    fn test_transform_work_items_preserves_existing_fields() {
        let work_items = json!({
            "workItems": {
                "nodes": [
                    {
                        "id": "1",
                        "title": "Lorem ipsum",
                        "existingField": "should be preserved",
                        "widgets": [
                            { "__typename": "WorkItemWidgetLabels", "type": "LABELS", "labels": mock_labels1() },
                        ],
                    },
                ]
            }
        });
        let work_items_data = work_items["workItems"].clone();
        let result = transform_work_items(&work_items_data);

        let expected = json!({
            "nodes": [
                {
                    "id": "1",
                    "title": "Lorem ipsum",
                    "existingField": "should be preserved",
                    "labels": mock_labels1(),
                    "widgets": JsonValue::Null,
                },
            ]
        });
        assert_eq!(result.unwrap(), expected);
    }

    #[test]
    fn test_transform_work_items_handles_malformed_data() {
        // Test with missing nodes
        let work_items = json!({
            "workItems": {
                "otherField": "value"
            }
        });
        let work_items_data = work_items["workItems"].clone();
        let result = transform_work_items(&work_items_data);
        assert_eq!(result.unwrap(), work_items["workItems"]);

        // Test with non-array nodes
        let work_items = json!({
            "workItems": {
                "nodes": "not an array"
            }
        });
        let work_items_data = work_items["workItems"].clone();
        let result = transform_work_items(&work_items_data);
        assert_eq!(result.unwrap(), work_items["workItems"]);
    }
}
