use crate::types::{Context, Field::*, Query, Source, Source::*, Value::*};
use crate::utils::feature_flags::FeatureFlag;

pub fn remove_extracted_fields(query: &mut Query, context: &Context) {
    query.expressions.retain(|expr| match &expr.field {
        Type => context.source == Some(Issues),
        Group | Project => false,
        _ => true,
    });
}

pub fn extract_context(query: &Query, context: &mut Context) {
    for expr in &query.expressions {
        match (&expr.field, &expr.value) {
            (Type, Token(value)) => {
                if value == "EPIC" {
                    // force enable glql work items for epics (except for subqueries)
                    // because the work items API is stable for epics with constraint
                    // excludeProjects: true, which is passed in graphql_filters.rs
                    FeatureFlag::GlqlWorkItems
                        .set(FeatureFlag::GlqlWorkItems.get() || !context.is_subquery);
                }
                context.source = Some(Source::from(value.clone()));
            }
            (Group, Quoted(value)) => {
                context.group = Some(value.clone());
                context.project = None;
            }
            (Project, Quoted(value)) => {
                context.project = Some(value.clone());
                context.group = None;
            }
            _ => {}
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::types::{Expression, Operator::*};

    #[test]
    fn test_extract_context() {
        let query = Query::new(vec![
            Expression::new(Group, Equal, Quoted("test-group".into())),
            Expression::new(Project, Equal, Quoted("test-project".into())),
            Expression::new(UnknownField("status".into()), Equal, Quoted("open".into())),
        ]);

        let mut context = Context {
            group: Some("gitlab-group".to_string()),
            ..Context::default()
        };

        extract_context(&query, &mut context);

        assert_eq!(context.project, Some("test-project".to_string()));
        // group is nullified
        assert_eq!(context.group, None);
    }

    #[test]
    fn test_extract_context_with_only_group() {
        let mut query = Query::new(vec![
            Expression::new(Group, Equal, Quoted("test-group".to_string())),
            Expression::new(Status, Equal, Quoted("open".to_string())),
        ]);

        let mut context = Context {
            project: Some("gitlab-project".to_string()),
            ..Context::default()
        };

        extract_context(&query, &mut context);
        remove_extracted_fields(&mut query, &context);

        assert_eq!(context.group, Some("test-group".to_string()));
        // preferred project is nullified
        assert_eq!(context.project, None);
        assert_eq!(query.expressions.len(), 1);
        assert_eq!(query.expressions[0].field, "status".into());
    }

    #[test]
    fn test_extract_context_with_no_group_or_project() {
        let mut query = Query::new(vec![
            Expression::new(
                UnknownField("status".into()),
                Equal,
                Quoted("open".to_string()),
            ),
            Expression::new(
                UnknownField("assignee".into()),
                Equal,
                Quoted("user1".to_string()),
            ),
        ]);

        let mut context = Context::default();

        extract_context(&query, &mut context);
        remove_extracted_fields(&mut query, &context);

        assert_eq!(context.group, None);
        assert_eq!(context.project, None);
        assert_eq!(query.expressions.len(), 2);
    }

    #[test]
    fn test_extract_context_with_non_string_values() {
        let mut query = Query::new(vec![
            Expression::new(Group, Equal, Number(123)),
            Expression::new(Group, Equal, Bool(true)),
        ]);

        let mut context = Context::default();

        extract_context(&query, &mut context);
        remove_extracted_fields(&mut query, &context);

        assert_eq!(context.group, None);
        assert_eq!(context.project, None);
        assert_eq!(query.expressions.len(), 0);
    }

    #[test]
    fn test_extract_context_with_type_issue() {
        let mut query = Query::new(vec![
            Expression::new(Type, Equal, Token("issue".to_string())),
            Expression::new(State, Equal, Token("opened".to_string())),
        ]);

        let mut context = Context::default();

        extract_context(&query, &mut context);
        remove_extracted_fields(&mut query, &context);

        // type param is retained
        assert_eq!(context.source, Some(Issues));
        assert_eq!(
            query.expressions,
            vec![
                Expression::new(Type, Equal, Token("issue".to_string())),
                Expression::new(State, Equal, Token("opened".to_string())),
            ]
        );
    }

    #[test]
    fn test_extract_context_with_type_merge_request() {
        let mut query = Query::new(vec![
            Expression::new(Type, Equal, Token("mergerequest".to_string())),
            Expression::new(State, Equal, Token("opened".to_string())),
        ]);

        let mut context = Context::default();

        extract_context(&query, &mut context);
        remove_extracted_fields(&mut query, &context);

        // type param is extracted
        assert_eq!(context.source, Some(MergeRequests));
        assert_eq!(
            query.expressions,
            vec![Expression::new(State, Equal, Token("opened".to_string())),]
        );
    }

    #[test]
    fn test_extract_context_with_type_and_project() {
        let mut query = Query::new(vec![
            Expression::new(Type, Equal, Token("mergerequest".to_string())),
            Expression::new(Project, Equal, Quoted("test-project".to_string())),
            Expression::new(State, Equal, Token("opened".to_string())),
        ]);

        let mut context = Context::default();

        extract_context(&query, &mut context);
        remove_extracted_fields(&mut query, &context);

        assert_eq!(context.source, Some(MergeRequests));
        assert_eq!(
            query.expressions,
            vec![Expression::new(State, Equal, Token("opened".to_string())),]
        );
    }
}
