use super::helpers::parse_reference;
use crate::errors::{GlqlError, GlqlErrorKind::*};
use crate::types::{
    ReferenceType::*,
    Value::{self, *},
};
use crate::utils::date_time::{TimeUnit, TimeUnit::*};
use nom::{
    Err, IResult,
    bytes::complete::{tag, take_while, take_while1},
    character::complete::{alpha1, alphanumeric1, digit1, one_of},
    combinator::{map, opt, recognize},
    sequence::pair,
};

// Parser for relative time
pub fn relative_date_inner(
    allow_negative: bool,
) -> impl Fn(&str) -> IResult<&str, (i64, TimeUnit), GlqlError> {
    move |input: &str| {
        let (input, sign) = opt(one_of(if allow_negative { "+-" } else { "+" }))(input)?;
        let (input, number) = digit1(input)?;
        let (input, unit) = alpha1(input)?;

        let sign = if let Some('-') = sign { -1 } else { 1 };
        let time_unit = match unit {
            "d" => Days,
            "w" => Weeks,
            "m" => Months,
            "y" => Years,
            input => {
                return Err(nom::Err::Failure(GlqlError::parse_error(
                    input,
                    InvalidRelativeDateUnit,
                )));
            }
        };

        Ok((input, (sign * number.parse::<i64>().unwrap(), time_unit)))
    }
}

pub fn relative_date(input: &str) -> IResult<&str, Value, GlqlError> {
    map(relative_date_inner(true), |(number, unit)| {
        RelativeDate(number, unit)
    })(input)
}

// Parser for dates
pub fn date(input: &str) -> IResult<&str, Value, GlqlError> {
    map(
        recognize(pair(
            digit1,
            pair(
                one_of("-/."),
                opt(pair(digit1, opt(pair(one_of("-/."), digit1)))),
            ),
        )),
        |s: &str| Date(s.to_string()),
    )(input)
}

// Parser for bare words
pub fn token(input: &str) -> IResult<&str, Value, GlqlError> {
    let (input, name) = alphanumeric1(input)?;
    match name.to_uppercase().as_str() {
        "ANY" | "NONE" | "CURRENT" | "UPCOMING" | "STARTED" | "ALL" | "OPENED" | "CLOSED"
        | "MERGED" | "ISSUE" | "MERGEREQUEST" | "INCIDENT" | "TESTCASE" | "REQUIREMENT"
        | "TASK" | "TICKET" | "OBJECTIVE" | "KEYRESULT" | "EPIC" => {
            Ok((input, Token(name.to_uppercase())))
        }
        _ => Err(Err::Failure(GlqlError::parse_error(name, InvalidTerm))),
    }
}

// Parser for usernames
pub fn username(input: &str) -> IResult<&str, Value, GlqlError> {
    let matcher = |s: &str| {
        s.chars().all(|c| c.is_alphanumeric() || "_.-".contains(c))
            && !s.chars().all(char::is_numeric)
    };
    parse_reference(UserRef, matcher, InvalidUsername, true)(input)
}

// Parser for milestone
pub fn milestone(input: &str) -> IResult<&str, Value, GlqlError> {
    let matcher = |s: &str| s.chars().all(|c| c.is_alphanumeric() || "_.-".contains(c));
    parse_reference(MilestoneRef, matcher, InvalidMilestone, true)(input)
}

// Parser for label
pub fn label(input: &str) -> IResult<&str, Value, GlqlError> {
    let matcher = |s: &str| s.chars().all(|c| c.is_alphanumeric() || "_.-:".contains(c));
    parse_reference(LabelRef, matcher, InvalidLabel, true)(input)
}

// Parser for iteration
pub fn iteration(input: &str) -> IResult<&str, Value, GlqlError> {
    let matcher = |s: &str| s.chars().all(char::is_numeric);
    parse_reference(IterationRef, matcher, InvalidIteration, false)(input)
}

// Parser for epic references
// Format: [group_path]&<epic_id>
// Examples:
//   &123 - Epic with ID 123 in the current context
//   gitlab-org&123 - Epic with ID 123 in the gitlab-org group
//
// Constraints:
// - Path can only contain alphanumeric characters, hyphens, and forward slashes
// - ID must be numeric
pub fn epic(input: &str) -> IResult<&str, Value, GlqlError> {
    let (input, path) = take_while(|c: char| c.is_alphanumeric() || "/-_.".contains(c))(input)?;
    let (input, _) = tag("&")(input)?;
    let (input, id) = take_while1(|c: char| c.is_numeric())(input)?;

    let reference = if path.is_empty() {
        id.to_string()
    } else {
        format!("{path}&{id}")
    };

    Ok((input, Reference(EpicRef, reference)))
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::parser::helpers::test_helpers::*;

    #[test]
    fn test_relative_date() {
        test_parser_result(relative_date, "-7d", RelativeDate(-7, Days));
        test_parser_result(relative_date, "+30m", RelativeDate(30, Months));
        // + is optional
        test_parser_result(relative_date, "12m", RelativeDate(12, Months));
        test_parser_error(relative_date, "7x");
    }

    #[test]
    fn test_date() {
        test_parser_result(date, "2023-05-01", Date("2023-05-01".to_string()));
        test_parser_error(date, "2023_05_01");
        test_parser_error(date, "");
        test_parser_error(date, "invalid");
    }

    #[test]
    fn test_token() {
        test_parser_result(token, "ANY", Token("ANY".to_string()));
        test_parser_result(token, "none", Token("NONE".to_string()));
        test_parser_error(token, "invalid");
    }

    #[test]
    fn test_username() {
        test_parser_result(
            username,
            "@user123",
            Reference(UserRef, "user123".to_string()),
        );
        test_parser_error(username, "user123");
    }

    #[test]
    fn test_milestone() {
        test_parser_result(
            milestone,
            "%17.5",
            Reference(MilestoneRef, "17.5".to_string()),
        );
        test_parser_result(
            milestone,
            "%\"17.5\"",
            Reference(MilestoneRef, "17.5".to_string()),
        );
        test_parser_result(
            milestone,
            "%\"Awaiting Further Demand\"",
            Reference(MilestoneRef, "Awaiting Further Demand".to_string()),
        );
        test_parser_result(
            milestone,
            "%Backlog",
            Reference(MilestoneRef, "Backlog".to_string()),
        );
        test_parser_result(
            milestone,
            "%\"Next 1-3 Releases\"",
            Reference(MilestoneRef, "Next 1-3 Releases".to_string()),
        );
        test_parser_result(
            milestone,
            "%feature-123",
            Reference(MilestoneRef, "feature-123".to_string()),
        );
        test_parser_result(
            milestone,
            "%\"  trims spaces  \"",
            Reference(MilestoneRef, "trims spaces".to_string()),
        );

        test_parser_error(milestone, "");
        test_parser_error(milestone, "%");
        test_parser_error(milestone, "%\"\"");
        test_parser_error(milestone, "17.5");

        test_parser_error(milestone, "Backlog");
        test_parser_error(milestone, "%\"Awaiting Further Demand");
        test_parser_error(milestone, "invalid milestone");
    }

    #[test]
    fn test_label() {
        test_parser_result(label, "~\"🌭\"", Reference(LabelRef, "🌭".to_string()));
        test_parser_result(
            label,
            "~\"1st contribution\"",
            Reference(LabelRef, "1st contribution".to_string()),
        );
        test_parser_result(label, "~AI", Reference(LabelRef, "AI".to_string()));
        test_parser_result(
            label,
            "~frontend",
            Reference(LabelRef, "frontend".to_string()),
        );
        test_parser_result(
            label,
            "~\"workflow::in review\"",
            Reference(LabelRef, "workflow::in review".to_string()),
        );
        test_parser_result(
            label,
            "~\"  trims spaces  \"",
            Reference(LabelRef, "trims spaces".to_string()),
        );

        test_parser_error(label, "");
        test_parser_error(label, "~");
        test_parser_error(label, "~\"\"");
        test_parser_error(label, "AI");
        test_parser_error(label, "~\"technical debt");
        test_parser_error(label, "~\"workflow::in dev");
    }

    #[test]
    fn test_iteration() {
        test_parser_result(
            iteration,
            "*iteration:25263",
            Reference(IterationRef, "25263".to_string()),
        );
        test_parser_result(
            iteration,
            "*iteration:1",
            Reference(IterationRef, "1".to_string()),
        );
        test_parser_error(iteration, "*iteration:");
        test_parser_error(iteration, "*iteration");
        test_parser_error(iteration, "*iteration:abc");
        test_parser_error(iteration, "*iteration:123abc");

        test_parser_error(iteration, "");
        test_parser_error(iteration, "iteration:25263");
    }

    #[test]
    fn test_epic() {
        test_parser_result(
            epic,
            "path/to/project-with_hyphens-underscores.and.dots&123",
            Reference(
                EpicRef,
                "path/to/project-with_hyphens-underscores.and.dots&123".to_string(),
            ),
        );
        test_parser_result(epic, "&123", Reference(EpicRef, "123".to_string()));
        test_parser_error(epic, "");
        test_parser_error(epic, "&");
        test_parser_error(epic, "&\"\"");
        test_parser_error(epic, "123");
        test_parser_error(epic, "path/to/project with spaces");
        test_parser_error(epic, "path/to/project-without-id");
    }
}
