use crate::errors::{GlqlError, GlqlErrorKind::*};
use crate::parser::{
    expression::expression,
    helpers::{ws0, ws1},
};
use crate::types::Query;
use nom::{IResult, bytes::complete::tag_no_case, multi::separated_list1};

mod dimensions;
mod expression;
mod fields;
mod helpers;
mod literals;
mod special;
mod value;

// Parser for the entire query (multiple expressions)
fn query(input: &str) -> IResult<&str, Query, GlqlError> {
    let (remaining, expressions) =
        ws0(separated_list1(ws1(tag_no_case("AND")), expression))(input)?;
    Ok((remaining, Query { expressions }))
}

pub use dimensions::parse_dimensions;
pub use fields::parse_fields;

pub fn parse_query(input: &str) -> Result<Query, GlqlError> {
    match query(input) {
        Ok(("", query)) => Ok(query),
        Ok((remaining, _)) => Err(GlqlError::parse_error(remaining, UnexpectedToken)),
        Err(nom::Err::Error(e) | nom::Err::Failure(e)) => Err(e),
        Err(_) => Err(GlqlError::parse_error("", UnknownError)),
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::types::{Expression, Field::*, Operator::*, Value::*};

    #[test]
    fn test_parse_query() {
        assert_eq!(
            parse_query("label = \"backend\" AND weight > 5"),
            Ok(Query {
                expressions: vec![
                    Expression::new(Label, Equal, Quoted("backend".into())),
                    Expression::new(Weight, GreaterThan, Number(5)),
                ]
            })
        );
        assert!(parse_query("invalid input").is_err());
    }
}
