use super::{helpers::ws0, value::non_array_value};
use crate::errors::{GlqlError, GlqlErrorKind::*};
use crate::types::{Value, Value::*};
use nom::{
    IResult,
    branch::alt,
    bytes::complete::{escaped, is_not, tag_no_case},
    character::complete::{alpha1, char, digit1, multispace0},
    combinator::map,
    multi::separated_list0,
    sequence::{delimited, preceded, terminated},
};

// Parser for null
pub fn null(input: &str) -> IResult<&str, Value, GlqlError> {
    map(tag_no_case("null"), |_| Null)(input)
}

// Parser for boolean
pub fn bool(input: &str) -> IResult<&str, Value, GlqlError> {
    alt((
        map(tag_no_case("true"), |_| Bool(true)),
        map(tag_no_case("false"), |_| Bool(false)),
    ))(input)
}

// Parser for numbers
pub fn number(input: &str) -> IResult<&str, Value, GlqlError> {
    map(digit1, |s: &str| Number(s.parse().unwrap()))(input)
}

fn string_parser(quote: char) -> impl Fn(&str) -> IResult<&str, Value, GlqlError> {
    move |input: &str| {
        let original_input = input;
        let (input, parsed_quote): (&str, char) = char::<&str, GlqlError>(quote)(input)?;
        let result = terminated(
            escaped(is_not(&format!("\\{quote}")[..]), '\\', char(quote)),
            char::<&str, GlqlError>(quote),
        )(input);

        match result {
            Ok((input, content)) => Ok((input, Quoted(content.trim().into()))),
            // starting quote was parsed successfully but rest of the string was not
            Err(_) if quote == parsed_quote => Err(nom::Err::Failure(GlqlError::parse_error(
                original_input,
                UnterminatedString,
            ))),
            Err(e) => Err(e),
        }
    }
}

// Parser for strings
pub fn string(input: &str) -> IResult<&str, Value, GlqlError> {
    alt((string_parser('"'), string_parser('\'')))(input)
}

// Parser for arrays
pub fn array(input: &str) -> IResult<&str, Value, GlqlError> {
    map(
        delimited(
            char('('),
            separated_list0(char(','), ws0(non_array_value)),
            char(')'),
        ),
        List,
    )(input)
}

// Parser for functions
pub fn function(input: &str) -> IResult<&str, Value, GlqlError> {
    let (input, name) = ws0(alpha1)(input)?;
    let (input, args) = delimited(
        ws0(char('(')),
        separated_list0(
            ws0(char(',')),
            map(string, |s| match s {
                Quoted(s) => s,
                _ => "".to_string(),
            }),
        ),
        preceded(multispace0, char(')')),
    )(input)?;

    Ok((input, Function(name.to_string(), args)))
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::parser::helpers::test_helpers::*;

    #[test]
    fn test_null() {
        test_parser_result(null, "null", Null);
        test_parser_error(null, "notnull");
    }

    #[test]
    fn test_bool() {
        test_parser_result(bool, "true", Bool(true));
        test_parser_result(bool, "false", Bool(false));
        test_parser_error(bool, "notbool");
    }

    #[test]
    fn test_number() {
        test_parser_result(number, "123 rest", Number(123));
        test_parser_error(number, "abc");
    }

    #[test]
    fn test_string() {
        test_parser_result(string, "\"hello\" rest", Quoted("hello".to_string()));
        test_parser_result(string, "'hello' rest", Quoted("hello".to_string()));
        test_parser_result(
            string,
            "\"  trims spaces  \"",
            Quoted("trims spaces".to_string()),
        );
        test_parser_result(
            string,
            "'  trims spaces  ' rest",
            Quoted("trims spaces".to_string()),
        );
        test_parser_error(string, "notstring");
    }

    #[test]
    fn test_array() {
        test_parser_result(
            array,
            "(\"a\", \"b\")",
            List(vec![Quoted("a".to_string()), Quoted("b".to_string())]),
        );
        test_parser_error(array, "(1, 2,)");
    }

    #[test]
    fn test_function() {
        test_parser_result(
            function,
            "currentuser()",
            Function("currentuser".to_string(), vec![]),
        );
        test_parser_result(
            function,
            "func(\"arg\")",
            Function("func".to_string(), vec!["arg".to_string()]),
        );
        test_parser_result(
            function,
            "func(\"arg1\", \"arg2\")",
            Function(
                "func".to_string(),
                vec!["arg1".to_string(), "arg2".to_string()],
            ),
        );
        test_parser_error(function, "func");
    }
}
