use crate::errors::{GlqlError, GlqlErrorKind};
use crate::types::{ReferenceType, Value, Value::*};
use nom::{
    IResult,
    branch::alt,
    bytes::complete::{tag_no_case, take_while1},
    character::complete::{char, multispace0, multispace1},
    combinator::map,
    sequence::delimited,
};

pub fn ws0<'a, F, O>(inner: F) -> impl FnMut(&'a str) -> IResult<&'a str, O, GlqlError>
where
    F: 'a + FnMut(&'a str) -> IResult<&'a str, O, GlqlError>,
{
    delimited(multispace0, inner, multispace0)
}

pub fn ws1<'a, F, O>(inner: F) -> impl FnMut(&'a str) -> IResult<&'a str, O, GlqlError>
where
    F: 'a + FnMut(&'a str) -> IResult<&'a str, O, GlqlError>,
{
    delimited(multispace1, inner, multispace1)
}

pub fn parse_reference(
    reference_type: ReferenceType,
    matcher: impl Fn(&str) -> bool,
    error_kind: GlqlErrorKind,
    allow_quoted: bool,
) -> impl Fn(&str) -> IResult<&str, Value, GlqlError> {
    let prefix = reference_type.symbol();

    move |input: &str| {
        let original_input = input;
        let (input, parsed_prefix) = tag_no_case(prefix.as_str())(input)?;

        // unquoted reference - match everything except whitespace, commas and parentheses
        let mut parser = map(
            take_while1(|c: char| !"(), \r\n\t".contains(c)),
            // is_quoted: false
            |s| (s, false),
        );
        let result: IResult<&str, (&str, bool), GlqlError> = if allow_quoted {
            alt((
                // quoted reference - any characters
                map(
                    delimited(char('"'), take_while1(|c: char| c != '"'), char('"')),
                    // is_quoted: true
                    |s| (s, true),
                ),
                parser,
            ))(input)
        } else {
            parser(input)
        };

        match result {
            // parse began with the correct prefix, but ended up with an error
            Err(_) if parsed_prefix == prefix => Err(nom::Err::Failure(GlqlError::parse_error(
                original_input,
                error_kind.clone(),
            ))),
            // apply matcher only if is_quoted: false
            Ok((_, (input, false))) if input.is_empty() || !matcher(input) => Err(
                nom::Err::Failure(GlqlError::parse_error(original_input, error_kind.clone())),
            ),
            Ok((input, (reference, _))) => Ok((
                input,
                Reference(reference_type.clone(), reference.trim().into()),
            )),
            Err(e) => Err(e),
        }
    }
}

#[cfg(test)]
pub mod test_helpers {
    use super::*;

    use std::fmt::Debug;

    // remove once we use this
    #[allow(dead_code)]
    pub fn test_parser<F, T>(func: F, input: &str)
    where
        F: Fn(&str) -> IResult<&str, T>,
        T: Debug + PartialEq,
    {
        match func(input) {
            Ok(_res) => {}
            Err(e) => {
                panic!("{e}")
            }
        }
    }

    pub fn test_parser_result<F, T>(func: F, input: &str, value: T)
    where
        F: for<'a> Fn(&'a str) -> IResult<&'a str, T, GlqlError>,
        T: Debug + PartialEq,
    {
        match func(input) {
            Ok(res) => {
                assert_eq!(value, res.1)
            }
            Err(e) => {
                panic!("{e}")
            }
        }
    }

    pub fn test_parser_error<F, T>(func: F, input: &str)
    where
        F: for<'a> Fn(&'a str) -> IResult<&'a str, T, GlqlError>,
        T: Debug + PartialEq,
    {
        match func(input) {
            Ok(res) => {
                println!("Unexpected success: {:?} parses into {:?}\n", input, res.1);
                panic!()
            }
            Err(_e) => {}
        }
    }
}
