use super::{helpers::ws0, literals::function};
use crate::errors::{GlqlError, GlqlErrorKind::*};
use crate::parser::expression::field;
use crate::parser::literals::string;
use crate::types::{
    DisplayField::{self, *},
    Value::*,
};
use nom::bytes::complete::tag_no_case;
use nom::sequence::preceded;
use nom::{
    IResult, branch::alt, character::complete::char, combinator::map, multi::separated_list0,
};

const FIELDS_SEPARATOR: char = ',';

fn static_field(input: &str) -> IResult<&str, DisplayField, GlqlError> {
    map(field, |field| Static(field.clone()))(input)
}

fn field_function(input: &str) -> IResult<&str, DisplayField, GlqlError> {
    function(input).and_then(|(input, value)| match value {
        Function(name, args) => Ok((input, FieldFunction(name, args))),
        _ => Err(nom::Err::Failure(GlqlError::parse_error(
            input,
            UnexpectedToken,
        ))),
    })
}

pub fn display_field(input: &str) -> IResult<&str, DisplayField, GlqlError> {
    let (input, field) = alt((field_function, static_field))(input)?;

    let trimmed_input = input.trim();
    if trimmed_input.is_empty() || trimmed_input.starts_with(FIELDS_SEPARATOR) {
        return Ok((input, field));
    }

    let err = |input: &str| {
        Err(nom::Err::Failure(GlqlError::parse_error(
            input,
            UnexpectedToken,
        )))
    };

    match preceded(ws0(tag_no_case("as")), string)(input) {
        Ok((input, alias)) => match alias {
            Quoted(alias) => Ok((input, field.aliased_as(alias.as_str()))),
            _ => err(input),
        },
        Err(_) => err(input),
    }
}

fn field_list(input: &str) -> IResult<&str, Vec<DisplayField>, GlqlError> {
    let (input, fields) = separated_list0(ws0(char(FIELDS_SEPARATOR)), display_field)(input)?;

    if fields.is_empty() {
        return Err(nom::Err::Failure(GlqlError::parse_error(
            input,
            MissingField,
        )));
    }

    Ok((input, fields))
}

pub fn parse_fields(input: &str) -> Result<Vec<DisplayField>, GlqlError> {
    match ws0(field_list)(input) {
        Ok(("", fields)) => Ok(fields),
        Ok((remaining, _)) => Err(GlqlError::parse_error(remaining, UnexpectedToken)),
        Err(nom::Err::Error(e) | nom::Err::Failure(e)) => Err(e),
        Err(_) => Err(GlqlError::parse_error("", UnknownError)),
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::types::Field::*;

    #[test]
    fn test_parse_single_field() {
        assert_eq!(parse_fields("title"), Ok(vec![Static(Title)]));
        assert_eq!(parse_fields("  title  "), Ok(vec![Static(Title)]));
    }

    #[test]
    fn test_parse_multiple_fields() {
        assert_eq!(
            parse_fields("title,description,createdAt"),
            Ok(vec![
                Static(Title),
                Static(Description),
                Static(Created.aliased_as("createdAt"))
            ])
        );
    }

    #[test]
    fn test_parse_function_call() {
        assert_eq!(
            parse_fields("labels('bug')"),
            Ok(vec![FieldFunction("labels".into(), vec!["bug".into()])])
        );
    }

    #[test]
    fn test_parse_function_call_with_multiple_arguments() {
        assert_eq!(
            parse_fields("labels('bug', 'feature')"),
            Ok(vec![FieldFunction(
                "labels".into(),
                vec!["bug".into(), "feature".into()]
            )])
        );
    }

    #[test]
    fn test_parse_mix_of_field_names_and_function_calls() {
        assert_eq!(
            parse_fields("title,labels('bug'),description"),
            Ok(vec![
                Static(Title),
                FieldFunction("labels".into(), vec!["bug".into()]),
                Static(Description),
            ])
        );
    }

    #[test]
    fn test_handle_whitespace() {
        assert_eq!(
            parse_fields(" title , labels( \"bug\" ) , description "),
            Ok(vec![
                Static(Title),
                FieldFunction("labels".into(), vec!["bug".into()]),
                Static(Description),
            ])
        );
    }

    #[test]
    fn test_handle_alias() {
        assert_eq!(
            parse_fields("title as \"t\", labels(\"bug\") as \"l\", description as \"d\""),
            Ok(vec![
                Static(Title).aliased_as("t"),
                FieldFunction("labels".into(), vec!["bug".into()]).aliased_as("l"),
                Static(Description).aliased_as("d"),
            ])
        );
    }

    #[test]
    fn test_handle_case_sensitive_alias() {
        assert_eq!(
            parse_fields("title As \"t\", description, createdAt AS \"c\""),
            Ok(vec![
                Static(Title).aliased_as("t"),
                Static(Description),
                Static(Created.aliased_as("createdAt")).aliased_as("c"),
            ])
        );
    }

    #[test]
    fn test_handle_alias_with_whitespace() {
        assert_eq!(
            parse_fields(
                " title  as  \"t\" , labels( \"bug\" )  as     \"l\" ,   description as  \"d\"  "
            ),
            Ok(vec![
                Static(Title).aliased_as("t"),
                FieldFunction("labels".into(), vec!["bug".into()]).aliased_as("l"),
                Static(Description).aliased_as("d"),
            ])
        );
    }

    #[test]
    fn test_error_scenarios() {
        assert_eq!(
            parse_fields("title as"),
            Err(GlqlError::parse_error(" as", UnexpectedToken))
        );

        assert_eq!(
            parse_fields("title as 123"),
            Err(GlqlError::parse_error(" as 123", UnexpectedToken))
        );

        assert_eq!(
            parse_fields("title,"),
            Err(GlqlError::parse_error(",", UnexpectedToken))
        );

        assert_eq!(
            parse_fields("labels('bug'"),
            Err(GlqlError::parse_error("('bug'", UnexpectedToken))
        );
    }
}
