use super::{helpers::ws0, literals::function, value::value};
use crate::errors::{GlqlError, GlqlErrorKind::*};
use crate::types::{Expression, Field, Field::*, Operator, Operator::*, Value::*};
use nom::{
    IResult,
    branch::alt,
    bytes::complete::{tag, tag_no_case},
    character::complete::{alpha1, char, multispace0, multispace1},
    combinator::map,
    sequence::{pair, terminated},
};

pub fn field(input: &str) -> IResult<&str, Field, GlqlError> {
    let static_field = map(alpha1, Field::from);

    alt((custom_field, static_field))(input)
}

pub fn custom_field(input: &str) -> IResult<&str, Field, GlqlError> {
    function(input).and_then(|(input, value)| match value {
        Function(name, args) if name.to_lowercase() != "customfield" => {
            Err(nom::Err::Failure(GlqlError::parse_error(
                input,
                UnrecognizedFunction {
                    name: name.clone(),
                    args: args.clone(),
                },
            )))
        }
        Function(name, args) if args.len() != 1 => Err(nom::Err::Failure(GlqlError::parse_error(
            input,
            UnexpectedNumberOfArguments {
                name: name.clone(),
                expected: 1,
                actual: args.len(),
            },
        ))),
        Function(_, args) => Ok((input, CustomField(args[0].clone()))),
        _ => Err(nom::Err::Failure(GlqlError::parse_error(
            input,
            UnexpectedToken,
        ))),
    })
}

// Parser for operators
fn operator(input: &str) -> IResult<&str, Operator, GlqlError> {
    alt((
        map(
            pair(tag_no_case("in"), multispace1::<&str, GlqlError>),
            |_| In,
        ),
        map(terminated(char('='), multispace0), |_| Equal),
        map(terminated(tag("!="), multispace0), |_| NotEqual),
        map(terminated(tag(">="), multispace0), |_| GreaterThanEquals),
        map(terminated(char('>'), multispace0), |_| GreaterThan),
        map(terminated(tag("<="), multispace0), |_| LessThanEquals),
        map(terminated(char('<'), multispace0), |_| LessThan),
    ))(input)
    .map_err(|_| nom::Err::Failure(GlqlError::parse_error(input, InvalidOperator)))
}

pub fn expression(input: &str) -> IResult<&str, Expression, GlqlError> {
    let original_input = input;

    // Check for missing field
    if input.is_empty() {
        return Err(nom::Err::Failure(GlqlError::parse_error(
            original_input,
            MissingField,
        )));
    }
    let (input, field) = field(input)?;

    // Check for missing operator
    if input.is_empty() {
        return Err(nom::Err::Failure(GlqlError::parse_error(
            original_input,
            MissingOperator,
        )));
    }
    let (input, operator) = ws0(operator)(input)?;

    // Check for missing value
    if input.is_empty() {
        return Err(nom::Err::Failure(GlqlError::parse_error(
            original_input,
            MissingValue,
        )));
    }
    let (input, value) = value(input)?;

    Ok((input, Expression::new(field, operator, value)))
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::parser::helpers::test_helpers::*;

    #[test]
    fn test_field() {
        test_parser_error(field, "123");
        test_parser_error(field, "");
        test_parser_result(field, "label rest", Label);
        test_parser_result(field, "thing other thing", UnknownField("thing".into()));
        test_parser_result(field, "here is some code", UnknownField("here".into()));
    }

    #[test]
    fn test_custom_field() {
        test_parser_result(
            field,
            "customField(\"Subscription\")",
            CustomField("Subscription".into()),
        );
        test_parser_error(field, "customField()");
        test_parser_error(field, "customField(\"Subscription\", \"Free\")");
        test_parser_error(field, "customField(\"Subscription\", \"Free\", \"Bar\")");
        test_parser_error(
            custom_field,
            "customField(\"Subscription\", \"Free\", \"Bar\")",
        );
    }

    #[test]
    fn test_operator() {
        test_parser_result(operator, "IN rest", In);
        test_parser_result(operator, "= rest", Equal);
        test_parser_result(operator, "!= rest", NotEqual);
        test_parser_result(operator, "> rest", GreaterThan);
        test_parser_result(operator, "< rest", LessThan);
        test_parser_result(operator, ">= rest", GreaterThanEquals);
        test_parser_result(operator, "<= rest", LessThanEquals);
        test_parser_error(operator, "invalid");
        test_parser_error(operator, "123");
        test_parser_error(operator, "");
    }

    #[test]
    fn test_expression() {
        test_parser_result(
            expression,
            "label = \"backend\"",
            Expression::new(Label, Equal, Quoted("backend".into())),
        );
        test_parser_error(expression, "label");
    }
}
