use super::{fields::display_field, helpers::ws0, special::relative_date_inner};
use crate::errors::{GlqlError, GlqlErrorKind::*};
use crate::types::aggregation::{Dimension, DimensionSegment};
use crate::utils::date_time::TimeSegmentType::*;
use nom::{
    IResult,
    bytes::complete::tag_no_case,
    character::complete::{alpha1, char},
    multi::separated_list0,
};

const DIMENSIONS_SEPARATOR: char = ',';

fn time_dimension_segment(input: &str) -> IResult<&str, DimensionSegment, GlqlError> {
    let (input, name) = ws0(alpha1)(input)?;
    if name.to_lowercase() != "timesegment" {
        return Err(nom::Err::Failure(GlqlError::parse_error(
            input,
            UnsupportedDimensionFunction {
                name: name.to_string(),
            },
        )));
    }

    let (input, _) = ws0(char('('))(input)?;
    let (input, relative_date) = relative_date_inner(false)(input)?;
    let (input, _) = ws0(char(')'))(input)?;

    Ok((
        input,
        DimensionSegment::TimeSegment(relative_date.0, relative_date.1, FromStartOfUnit),
    ))
}

fn dimension_segment(input: &str) -> IResult<&str, DimensionSegment, GlqlError> {
    time_dimension_segment(input)
}

fn dimension(input: &str) -> IResult<&str, Dimension, GlqlError> {
    // parse DIMENSION_SEGMENT on FIELD
    let (input, segment) = dimension_segment(input)?;
    // match "on"
    let (input, _) = ws0(tag_no_case("on"))(input)?;
    let (input, field) = display_field(input)?;

    Ok((input, Dimension { field, segment }))
}

fn dimension_list(input: &str) -> IResult<&str, Vec<Dimension>, GlqlError> {
    let (input, dimensions) = separated_list0(ws0(char(DIMENSIONS_SEPARATOR)), dimension)(input)?;

    if dimensions.is_empty() {
        return Err(nom::Err::Failure(GlqlError::parse_error(
            input,
            UnexpectedToken,
        )));
    }

    if dimensions.len() > 1 {
        return Err(nom::Err::Failure(GlqlError::parse_error(
            input,
            MaxDimensionsExceeded,
        )));
    }

    Ok((input, dimensions))
}

pub fn parse_dimensions(input: &str) -> Result<Vec<Dimension>, GlqlError> {
    match ws0(dimension_list)(input) {
        Ok(("", dimensions)) => Ok(dimensions),
        Ok((remaining, _)) => Err(GlqlError::parse_error(remaining, UnexpectedToken)),
        Err(nom::Err::Error(e) | nom::Err::Failure(e)) => Err(e),
        Err(_) => Err(GlqlError::parse_error("", UnknownError)),
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::types::{
        DisplayField::*,
        Field::*,
        aggregation::{Dimension, DimensionSegment::*},
    };
    use crate::utils::date_time::TimeUnit::*;

    #[test]
    fn test_valid_time_segment_dimension() {
        assert_eq!(
            parse_dimensions("timeSegment(1d) on created").unwrap(),
            vec![Dimension::segment(TimeSegment(1, Days, FromStartOfUnit)).on(Static(Created))]
        );

        assert_eq!(
            parse_dimensions("timeSegment(1d) on createdAt").unwrap(),
            vec![
                Dimension::segment(TimeSegment(1, Days, FromStartOfUnit))
                    .on(Static(Created.aliased_as("createdAt")))
            ]
        );

        assert_eq!(
            parse_dimensions("timeSegment(1m) on merged").unwrap(),
            vec![Dimension::segment(TimeSegment(1, Months, FromStartOfUnit)).on(Static(Merged))]
        );

        assert_eq!(
            parse_dimensions("timeSegment(1y) on closed").unwrap(),
            vec![Dimension::segment(TimeSegment(1, Years, FromStartOfUnit)).on(Static(Closed))]
        );

        assert_eq!(
            parse_dimensions("timeSegment(1d) on createdAt as 'Date created'").unwrap(),
            vec![
                Dimension::segment(TimeSegment(1, Days, FromStartOfUnit))
                    .on(Static(Created.aliased_as("createdAt")).aliased_as("Date created"))
            ]
        );
    }

    #[test]
    fn test_invalid_dimensions() {
        assert_eq!(
            parse_dimensions("timeSegment(1d)").unwrap_err().to_string(),
            "Error: Unexpected token at end of input: `timeSegment(1d)`"
        );

        assert_eq!(
            parse_dimensions("timeSegment(1d) on closed, timeSegment(1d) on created")
                .unwrap_err()
                .to_string(),
            "Error: Only one dimension is supported"
        );
    }
}
