pub mod analyzer;
pub mod codegen;
pub mod errors;
pub mod field_mapping;
pub mod parser;
pub mod transformer;
pub mod types;
pub mod utils;

use crate::{
    parser::parse_fields,
    types::{CompileContext, TransformContext, TransformOutput, aggregation::AggregationContext},
};
use analyzer::analyze;
use codegen::graphql::generate_code;
use errors::GlqlError;
use parser::parse_query;
use std::{fs, path::Path};
use transformer::{
    context::{extract_context, remove_extracted_fields},
    data::transform as transform_data,
    functions::apply_functions,
    transform as transform_query,
};
use types::{CompileOutput, Context};
use utils::{
    common::{reset_unique_id_counters, sha256},
    feature_flags::FeatureFlag,
};
use wasm_bindgen::prelude::*;

fn compile_glql_wrapped(query: &str, context: &mut Context) -> Result<CompileOutput, GlqlError> {
    let mut query = parse_query(query)?;
    apply_functions(&mut query, context)?;
    extract_context(&query, context);
    analyze(&query, context)?;
    remove_extracted_fields(&mut query, context);

    let transformed = transform_query(&query, context);

    Ok(generate_code(transformed, context))
}

pub fn compile_glql(query: &str, context: &mut Context) -> CompileOutput {
    let compiled = compile_glql_wrapped(query, context)
        .unwrap_or_else(|err| CompileOutput::error(&err.to_string()));

    if compiled.success && std::env::var("DUMP_GRAPHQL").is_ok() {
        let sha = sha256(&compiled.output);
        let dir = Path::new("tmp/graphql");
        fs::create_dir_all(dir).unwrap();

        let file_path = dir.join(format!("{}.graphql", &sha[..8]));
        fs::write(file_path, &compiled.output).unwrap();
    }

    compiled
}

#[wasm_bindgen]
pub fn compile(query: &str, string_context: &str) -> String {
    reset_unique_id_counters();

    let raw_context: Result<CompileContext, _> = serde_json::from_str(string_context);
    let raw_context = match raw_context {
        Ok(raw_context) => raw_context,
        Err(err) => {
            return CompileOutput::error(&err.to_string()).serialize();
        }
    };

    if let Some(feature_flags) = raw_context.feature_flags
        && let Some(glql_work_items) = feature_flags.glql_work_items
    {
        FeatureFlag::GlqlWorkItems.set(glql_work_items);
    }

    let fields = match parse_fields(raw_context.fields.as_str()) {
        Ok(fields) => fields,
        Err(err) => {
            return CompileOutput::error(&err.to_string()).serialize();
        }
    };

    let aggregate = if let Some(aggregate) = raw_context.aggregate {
        match AggregationContext::try_from(aggregate) {
            Ok(aggregate) => Some(aggregate),
            Err(err) => {
                return CompileOutput::error(&err.to_string()).serialize();
            }
        }
    } else {
        None
    };

    let mut context = Context {
        username: raw_context.username,
        fields,
        project: raw_context.project,
        group: raw_context.group,
        sort: raw_context.sort.map(|s| s.as_str().into()),
        aggregate,
        ..Context::default()
    };

    compile_glql(query, &mut context).serialize()
}

#[wasm_bindgen]
pub fn transform(data: &str, string_context: &str) -> String {
    let raw_context: Result<TransformContext, _> = serde_json::from_str(string_context);
    let raw_context = match raw_context {
        Ok(raw_context) => raw_context,
        Err(err) => {
            return TransformOutput::error(&err.to_string()).serialize();
        }
    };

    let fields = if let Some(aggregate) = raw_context.aggregate {
        match AggregationContext::try_from(aggregate) {
            Ok(aggregate) => {
                let mut aggregate_fields = aggregate
                    .dimensions
                    .iter()
                    .map(|d| d.field.clone())
                    .collect::<Vec<_>>();
                aggregate_fields.extend(aggregate.metrics.iter().cloned());
                aggregate_fields
            }
            Err(err) => {
                return TransformOutput::error(&err.to_string()).serialize();
            }
        }
    } else if let Some(f) = raw_context.fields {
        match parse_fields(f.as_str()) {
            Ok(fields) => fields,
            Err(err) => {
                return TransformOutput::error(&err.to_string()).serialize();
            }
        }
    } else {
        return TransformOutput::error("No fields provided").serialize();
    };

    let data = match serde_json::from_str(data) {
        Ok(data) => data,
        Err(err) => {
            return TransformOutput::error(&err.to_string()).serialize();
        }
    };

    match transform_data(&data, &fields) {
        Ok(transformed) => TransformOutput::success(transformed, fields.clone()),
        Err(err) => TransformOutput::error(&err.to_string()),
    }
    .serialize()
}

#[cfg(test)]
mod tests {
    use super::*;
    use serde_json::json;

    #[test]
    fn test_fields_containing_functions() {
        assert!(
            compile(
                "type = Issue",
                json!({
                    "username": "test",
                    "fields": "labels(\"workflow::*\")"
                })
                .to_string()
                .as_str(),
            )
            .contains("labels")
        );
    }
}
