use crate::compile_glql;
use crate::types::{
    Context, Field, Field::*, FieldMapping, FieldType, FieldType::*, ReferenceType::*,
    RelationshipType::*, Source::*, Value,
};
use crate::utils::common::parse_epic_reference;
use crate::utils::feature_flags::FeatureFlag;

/// Attribute function for work items epic mapping (parent work items)
fn epic_work_items_attribute(_field_type: &FieldType, value: &Value) -> String {
    use crate::types::Value::*;

    // For work items, wildcard values (null/tokens) use wildcard attribute
    // All other values (including lists) use parentIds
    if matches!(value, Null | Token(_)) {
        "parentWildcardId".to_string()
    } else {
        "parentIds".to_string()
    }
}

/// Attribute function for legacy epic mapping
fn epic_legacy_attribute(field_type: &FieldType, value: &Value) -> String {
    use crate::types::Value::*;

    if field_type.is_nullable() && matches!(value, Null | Token(_)) {
        "epicWildcardId".to_string()
    } else {
        "epicId".to_string()
    }
}

/// Returns the field mapping configuration for a given field and context
/// This centralizes the logic for how fields map to different API implementations
pub fn get_field_mapping(field: &Field, context: &Context) -> Option<FieldMapping> {
    match (
        field.dealias(),
        FeatureFlag::GlqlWorkItems.get(),
        &context.source,
    ) {
        // Epic field with work items API (epics are treated as parent work items)
        (Field::Epic, true, Some(Issues)) => Some(FieldMapping {
            field_type: StringLike
                | NumberLike
                | ReferenceLike(EpicRef)
                | ListLike(
                    HasOne,
                    Box::new(StringLike | NumberLike | ReferenceLike(EpicRef)),
                ),
            variable_type: "WorkItemID!".to_string(),
            get_attribute: epic_work_items_attribute,
        }),

        // Epic field with legacy API
        (Field::Epic, false, _) => Some(FieldMapping {
            field_type: StringLike | NumberLike | Nullable | ReferenceLike(EpicRef),
            variable_type: "String".to_string(),
            get_attribute: epic_legacy_attribute,
        }),

        // Future: Parent field implementation would go here
        // (Field::Parent, true, Some(Issues)) => Some(FieldMapping {
        //     field_type: StringLike | NumberLike | ReferenceLike(WorkItemRef) |
        //                ListLike(HasOne, Box::new(StringLike | NumberLike | ReferenceLike(WorkItemRef))),
        //     variable_type: "WorkItemID!".to_string(),
        //     get_attribute: parent_work_items_attribute, // Different logic - no group derivation
        // }),
        _ => None,
    }
}

/// Transforms a value for work items epic references
/// Handles epic references like "&123" or "group&123" and generates work items API subqueries
pub fn transform_to_work_item_epic_subquery(v: &Value, context: &mut Context) -> Value {
    transform_epic_subquery_with_variable_type(v, context, "WorkItemID!")
}

/// Transforms a value for legacy epic references
/// Handles epic references for legacy API and generates legacy epics API subqueries
pub fn transform_to_legacy_epic_subquery(v: &Value, context: &mut Context) -> Value {
    transform_epic_subquery_with_variable_type(v, context, "String")
}

/// Common epic subquery transformation logic with configurable variable type
fn transform_epic_subquery_with_variable_type(
    v: &Value,
    context: &mut Context,
    variable_type: &str,
) -> Value {
    use crate::types::{Value::*, Variable};
    use crate::utils::common::unique_id;
    use std::path::Path;

    let id = match v {
        Number(n) => n.to_string(),
        Quoted(s) => s.clone(),
        Reference(_, s) => s.clone(),
        List(arr) => {
            return List(
                arr.iter()
                    .map(|v| transform_epic_subquery_with_variable_type(v, context, variable_type))
                    .collect(),
            );
        }
        _ => return v.clone(),
    };

    let epic_group = match &context.group {
        Some(group) => group.clone(),
        None => {
            let project_path = context.project.clone().unwrap_or("".to_string());
            let path = Path::new(&project_path);
            if let Some(parent) = path.parent() {
                parent.to_string_lossy().to_string()
            } else {
                "".to_string()
            }
        }
    };

    let mut subquery_context = Context {
        fields: vec![Id.into()],
        is_subquery: true,
        group: context.group.clone(),
        project: context.project.clone(),
        ..Context::default()
    };

    let variable_key = unique_id("epicId");
    let default_query = |group: &str, id: &str| {
        format!("type = Epic AND id = {id} AND includeSubgroups = false AND group = \"{group}\"")
    };

    let subquery = |query| {
        context.variables.push(
            Variable::new(&variable_key, variable_type)
                .with_value(&compile_glql(query, &mut subquery_context).output),
        );

        Subquery(
            query.to_string(),
            Variable::new(&variable_key, variable_type),
            Box::new(subquery_context),
        )
    };

    // Parse the epic reference using the utility function
    let (group_path, epic_id) = parse_epic_reference(&id, &epic_group);
    subquery(&default_query(&group_path, &epic_id))
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::types::{Context, Field};

    #[test]
    fn test_get_field_mapping_epic_work_items() {
        let context = Context {
            source: Some(Issues),
            ..Context::default()
        };

        // Set the feature flag properly
        FeatureFlag::GlqlWorkItems.set(true);

        let mapping = get_field_mapping(&Field::Epic, &context).unwrap();
        assert_eq!(mapping.variable_type, "WorkItemID!");
        assert!(mapping.field_type.is_list_like());

        // Test attribute function
        use crate::types::Value::*;
        assert_eq!(
            (mapping.get_attribute)(&mapping.field_type, &Null),
            "parentWildcardId"
        );
        assert_eq!(
            (mapping.get_attribute)(&mapping.field_type, &Token("any".to_string())),
            "parentWildcardId"
        );
        assert_eq!(
            (mapping.get_attribute)(&mapping.field_type, &Number(123)),
            "parentIds"
        );
        assert_eq!(
            (mapping.get_attribute)(&mapping.field_type, &List(vec![])),
            "parentIds"
        );
    }

    #[test]
    fn test_get_field_mapping_epic_legacy() {
        let context = Context {
            source: Some(Issues),
            ..Context::default()
        };

        // Set the feature flag properly
        FeatureFlag::GlqlWorkItems.set(false);

        let mapping = get_field_mapping(&Field::Epic, &context).unwrap();
        assert_eq!(mapping.variable_type, "String");
        assert!(mapping.field_type.is_nullable());

        // Test attribute function
        use crate::types::Value::*;
        assert_eq!(
            (mapping.get_attribute)(&mapping.field_type, &Null),
            "epicWildcardId"
        );
        assert_eq!(
            (mapping.get_attribute)(&mapping.field_type, &Token("any".to_string())),
            "epicWildcardId"
        );
        assert_eq!(
            (mapping.get_attribute)(&mapping.field_type, &Number(123)),
            "epicId"
        );
    }

    #[test]
    fn test_get_field_mapping_unknown_field() {
        let context = Context::default();
        let mapping = get_field_mapping(&Field::Label, &context);
        assert!(mapping.is_none());
    }
}
