use crate::codegen::graphql_filters::to_type_param;
use crate::types::{
    Field::{self, *},
    FieldType, Operator, Source,
    Value::{self, *},
};
use GlqlErrorKind::*;
use heck::ToUpperCamelCase;
use nom::error::{ErrorKind, ParseError};
use std::fmt;

#[derive(Debug, PartialEq)]
pub struct GlqlError {
    pub input: Option<String>,
    pub code: Box<GlqlErrorKind>,
}

#[derive(Debug, Clone, PartialEq, Eq)]
pub enum GlqlErrorKind {
    UnknownError,
    InvalidRelativeDateUnit,
    InvalidTerm,
    UnterminatedString,
    MissingField,
    MissingOperator,
    MissingValue,
    InvalidOperator,
    InvalidUsername,
    InvalidMilestone,
    InvalidLabel,
    InvalidIteration,
    CannotCombineValues,
    InvalidEpic,

    ExpectedFieldToBeArray {
        value: String,
    },
    InvalidDateFormat {
        date: String,
    },
    UnrecognizedFunction {
        name: String,
        args: Vec<String>,
    },
    UnexpectedNumberOfArguments {
        name: String,
        expected: usize,
        actual: usize,
    },
    UnrecognizedField {
        field: String,
    },
    UnexpectedToken,
    SourceNotProvided,
    UnrecognizedFieldForSource {
        field: Field,
        source: Source,
    },
    UnrecognizedSortFieldForSource {
        field: Field,
        source: Source,
        supported_fields: Vec<Field>,
    },
    InvalidSortOrder {
        order: String,
    },
    DuplicateOperatorForField {
        field: Field,
    },
    UnsupportedOperatorForField {
        field: Field,
        operator: Operator,
        supported_operators: Vec<Operator>,
    },
    UnsupportedOperatorValueForField {
        field: Field,
        operator: Operator,
        value: Value,
        supported_operators: String,
    },
    UnsupportedValueForField {
        field: Field,
        value: Value,
        supported_value_types: Vec<FieldType>,
    },
    IncorrectFieldPairing {
        field: Field,
        supported_fields: Vec<Field>,
    },

    InvalidScope {
        scope_type: InvalidScopeType,
    },

    ArrayTooLarge {
        field: Field,
        max_allowed: usize,
    },
    UnsupportedDimensionFunction {
        name: String,
    },
    MaxDimensionsExceeded,
}

#[derive(Debug, Clone, PartialEq, Eq)]
pub enum InvalidScopeType {
    Project,
    Group,
    Namespace,
}

impl GlqlError {
    pub fn parse_error(input: &str, error: GlqlErrorKind) -> Self {
        GlqlError {
            input: Some(input.to_string()),
            code: Box::new(error),
        }
    }

    pub fn analyzer_error(error: GlqlErrorKind) -> Self {
        GlqlError {
            input: None,
            code: Box::new(error),
        }
    }
}

impl<I: fmt::Display> ParseError<I> for GlqlError {
    fn from_error_kind(input: I, _: ErrorKind) -> Self {
        GlqlError {
            input: Some(input.to_string()),
            code: Box::new(GlqlErrorKind::UnknownError),
        }
    }

    fn append(_: I, _: ErrorKind, other: Self) -> Self {
        other
    }
}

impl fmt::Display for GlqlError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let input = self.input.clone().unwrap_or_default();
        let error = match *self.code.clone() {
            UnknownError => format!("Unexpected token near `{input}`"),
            InvalidRelativeDateUnit => format!(
                "Unexpected character `{input}`. Expected d (day), w (week), m (month), or y (year)"
            ),
            InvalidTerm => format!("Unexpected term `{input}`"),
            UnterminatedString => format!("Unterminated string `{input}`"),
            MissingField => "Unknown error. Is there a string to parse?".to_string(),
            MissingOperator => {
                "Unexpected end of input, expected operator (one of IN, =, !=, >, or <)".to_string()
            }
            MissingValue => "Unexpected end of input, expected valid value".to_string(),
            InvalidOperator => {
                format!("Unexpected `{input}`, expected operator (one of IN, =, !=, >, or <)")
            }
            InvalidUsername => format!("Invalid username reference `{input}`"),
            InvalidMilestone => format!("Invalid milestone reference `{input}`"),
            InvalidLabel => format!("Invalid label reference `{input}`"),
            InvalidIteration => format!("Invalid iteration reference `{input}`"),
            CannotCombineValues => "Cannot combine these values".to_string(),
            ExpectedFieldToBeArray { value } => format!("Expected '{value}' to be an array"),
            InvalidDateFormat { date } => {
                format!("Invalid date format '{date}'. Expected format: yyyy-mm-dd")
            }
            UnrecognizedFunction { name, args } => format!(
                "Unrecognized function: {}({})",
                name,
                args.iter()
                    .map(|s| format!("\"{s}\""))
                    .collect::<Vec<_>>()
                    .join(", ")
            ),
            UnexpectedNumberOfArguments {
                name,
                expected,
                actual,
            } => format!(
                "Unexpected number of arguments for function: `{name}`. Expected: {expected}, Actual: {actual}"
            ),
            UnrecognizedField { field } => format!("{field} is not a recognized field."),
            UnexpectedToken => format!("Unexpected token at end of input: `{input}`"),
            SourceNotProvided => "Source is required".to_string(),
            UnrecognizedFieldForSource { field, source } => {
                format!("`{field}` is not a recognized field for {source}.")
            }
            UnrecognizedSortFieldForSource {
                field,
                source,
                supported_fields,
            } => {
                format!(
                    "`{}` is not a recognized sort field for {}. Supported fields to sort by: {}.",
                    field,
                    source,
                    supported_fields
                        .iter()
                        .map(|f| format!("`{f}`"))
                        .collect::<Vec<_>>()
                        .join(", ")
                )
            }
            InvalidSortOrder { order } => {
                format!("Invalid sort order: `{order}`. Valid sort orders: `asc`, `desc`.",)
            }
            DuplicateOperatorForField { field } => {
                format!("The is one of (`in`) operator can only be used once with `{field}`.")
            }
            UnsupportedOperatorForField {
                field,
                operator,
                supported_operators,
            } => format!(
                "`{}` does not support the {} operator. Supported operators: {}.",
                field,
                operator,
                supported_operators
                    .iter()
                    .map(ToString::to_string)
                    .collect::<Vec<_>>()
                    .join(", ")
            ),
            UnsupportedOperatorValueForField {
                field,
                operator,
                value,
                supported_operators,
            } => {
                let s_operators = if supported_operators.is_empty() {
                    "".to_string()
                } else {
                    format!(" Supported operators: {supported_operators}.")
                };

                format!(
                    "`{field}` does not support the {operator} operator for `{value}`.{s_operators}"
                )
            }
            UnsupportedValueForField {
                field,
                value,
                supported_value_types,
            } => {
                let mut message = format!(
                    "`{}` cannot be compared with `{}`. Supported value types: {}.",
                    field,
                    value,
                    supported_value_types
                        .iter()
                        .map(ToString::to_string)
                        .collect::<Vec<_>>()
                        .join(", "),
                );

                match (field.dealias(), &value) {
                    (Type, List(items))
                        if items.iter().any(|v| matches!(v, Token(s) if s == "EPIC")) =>
                    {
                        message = format!(
                            "Type `Epic` cannot be combined with other types ({}) in the same query. Try using `type = Epic` instead.",
                            items // without Epic
                                .iter()
                                .filter(|v| **v != Token("EPIC".to_string()))
                                .map(|v| format!(
                                    "`{}`",
                                    to_type_param(&v.to_string()).to_upper_camel_case()
                                ))
                                .collect::<Vec<_>>()
                                .join(", ")
                        );
                    }
                    (Type, List(items))
                        if items
                            .iter()
                            .any(|v| matches!(v, Token(s) if s == "MERGEREQUEST")) =>
                    {
                        message = format!(
                            "Type `MergeRequest` cannot be combined with other types ({}) in the same query. Try using `type = MergeRequest` instead.",
                            items // without MergeRequest
                                .iter()
                                .filter(|v| **v != Token("MERGEREQUEST".to_string()))
                                .map(|v| format!(
                                    "`{}`",
                                    to_type_param(&v.to_string()).to_upper_camel_case()
                                ))
                                .collect::<Vec<_>>()
                                .join(", ")
                        );
                    }
                    (field @ (Created | Closed), Bool(b)) => {
                        let is_created = matches!(field, Created);
                        let state = if is_created == *b { "opened" } else { "closed" };
                        message += &format!(" Did you mean `state = {state}`?");
                    }
                    _ => {}
                }

                message
            }
            IncorrectFieldPairing {
                field,
                supported_fields,
            } => format!(
                "`{}` can only be used with: {}.",
                field,
                supported_fields
                    .iter()
                    .map(|f| format!("`{f}`"))
                    .collect::<Vec<_>>()
                    .join(", ")
            ),
            InvalidEpic => format!("Invalid epic reference `{input}`"),
            InvalidScope { scope_type } => match scope_type {
                InvalidScopeType::Project => {
                    "Project does not exist or you do not have access to it".to_string()
                }
                InvalidScopeType::Group => {
                    "Group does not exist or you do not have access to it".to_string()
                }
                InvalidScopeType::Namespace => {
                    "Namespace does not exist or you do not have access to it".to_string()
                }
            },

            ArrayTooLarge { field, max_allowed } => {
                format!("`{field}` field exceeds maximum limit of {max_allowed} items.")
            }
            UnsupportedDimensionFunction { name } => {
                format!("`{name}` is not a supported dimension function.")
            }
            MaxDimensionsExceeded => "Only one dimension is supported".to_string(),
        };

        write!(f, "Error: {error}")
    }
}
