use crate::field_mapping::get_field_mapping;
use crate::types::{
    Context, Expression,
    Field::*,
    HealthStatus::{self, *},
    Operator::*,
    Query, Sort,
    Source::*,
    Value,
    Value::*,
    Variable,
};
use crate::utils::feature_flags::FeatureFlag;
use core::panic;
use indexmap::map::IndexMap;
use lazy_static::lazy_static;
use std::{
    convert::{From, TryFrom},
    fmt,
};

#[derive(Debug, Default, Clone)]
pub struct GraphQLFilters {
    pub and: IndexMap<String, Value>,
    pub or: IndexMap<String, Value>,
    pub not: IndexMap<String, Value>,
    pub constraints: IndexMap<String, Value>,
    pub custom_fields: IndexMap<String, Value>,
}

impl GraphQLFilters {
    pub fn new(query: &Query, context: &mut Context) -> Self {
        let mut filters = GraphQLFilters::default();
        let constraints = &mut filters.constraints;

        for expr in &query.expressions {
            let key = to_graphql_filter_key(expr, context);
            let value = to_graphql_filter_value(expr, context);

            if let CustomField(name) = expr.field.dealias() {
                filters.custom_fields.insert(name.clone(), value);
                continue;
            }

            match expr.operator {
                Equal | GreaterThan | LessThan | GreaterThanEquals | LessThanEquals => {
                    &mut filters.and
                }
                NotEqual => &mut filters.not,
                In => &mut filters.or,
            }
            .insert(key, value);
        }

        constraints.insert("before".to_string(), Token("$before".to_string()));
        constraints.insert("after".to_string(), Token("$after".to_string()));
        constraints.insert("first".to_string(), Token("$limit".to_string()));
        context.variables.push(Variable::new("before", "String"));
        context.variables.push(Variable::new("after", "String"));
        context.variables.push(Variable::new("limit", "Int"));

        if let Some(sort) = &context.sort {
            constraints.insert("sort".to_string(), Token(to_graphql_sort(sort)));
        }

        if context.group.is_some() && !query.contains_field(&IncludeSubgroups) {
            constraints.insert(
                match FeatureFlag::GlqlWorkItems.get() {
                    // work items only
                    true if context.source == Some(Issues) => "includeDescendants",
                    // legacy epics
                    false if context.source == Some(Epics) => "includeDescendantGroups",
                    // legacy issues and merge requests
                    _ => "includeSubgroups",
                }
                .to_string(),
                Bool(true),
            );
        }

        // exclude projects for epics for performance reasons
        if context.group.is_some()
            && query.contains_field_with_value(&Type, &Token("EPIC".to_string()))
        {
            constraints.insert("excludeProjects".to_string(), Bool(true));
        }

        if query.contains_field(&Epic) {
            constraints.insert(
                match FeatureFlag::GlqlWorkItems.get() {
                    true => "includeDescendantWorkItems".to_string(),
                    false => "includeSubepics".to_string(),
                },
                Bool(true),
            );
        }

        filters
    }
}

impl fmt::Display for GraphQLFilters {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let mut filters = String::new();

        fn format_value(value: &Value) -> String {
            match value {
                Null => "null".to_string(),
                Bool(b) => b.to_string(),
                Number(n) => n.to_string(),
                Token(s) => s.to_string(),
                Quoted(s) | Reference(_, s) | Date(s) => format!("\"{s}\""),
                List(arr) => format!(
                    "[{}]",
                    arr.iter().map(format_value).collect::<Vec<_>>().join(",")
                ),

                RelativeDate(..) | Function(..) => {
                    panic!("Error: Cannot generate code before function evaluation.")
                }
                Subquery(_, variable, _) => format!("${}", variable.key),
            }
        }

        // Helper function to format filters
        fn format_filters(filters: &IndexMap<String, Value>, wrapper: Option<&str>) -> String {
            if filters.is_empty() {
                return "".to_string();
            }
            let mut formatted = Vec::new();
            for (k, v) in filters {
                formatted.push(format!("{}: {}", k, format_value(v)));
            }
            let joined = formatted.join(" ");

            wrapper.map_or(format!("{joined} "), |w| format!("{w}: {{ {joined} }} "))
        }

        fn format_custom_field_filters(custom_fields: &IndexMap<String, Value>) -> String {
            if custom_fields.is_empty() {
                return "".to_string();
            }
            let mut formatted = vec![];
            for (name, value) in custom_fields {
                formatted.push(format!(
                    "{{customFieldName: \"{name}\", selectedOptionValues: {value}}}",
                    value = format_value(value)
                ));
            }
            format!("customField: [{}] ", formatted.join(" "))
        }

        filters += &format_filters(&self.and, None);
        filters += &format_filters(&self.or, Some("or"));
        filters += &format_filters(&self.not, Some("not"));
        filters += &format_filters(&self.constraints, None);
        filters += &format_custom_field_filters(&self.custom_fields);

        if filters.is_empty() {
            return write!(f, "{filters}");
        }

        write!(f, "({filters})")
    }
}

// Convert from HealthStatus to GraphQL enum string
impl From<HealthStatus> for String {
    fn from(status: HealthStatus) -> Self {
        match status {
            OnTrack => "onTrack".to_string(),
            NeedsAttention => "needsAttention".to_string(),
            AtRisk => "atRisk".to_string(),
        }
    }
}

// Convert from HealthStatus to Value
impl From<HealthStatus> for Value {
    fn from(status: HealthStatus) -> Self {
        Token(String::from(status))
    }
}

fn to_health_status(v: &Value) -> Value {
    match v {
        Quoted(s) => match HealthStatus::try_from(s.as_str()) {
            Ok(status) => Value::from(status),
            Err(_) => v.clone(),
        },
        _ => v.clone(),
    }
}

fn to_iteration_cadence_id(v: &Value) -> Value {
    match v {
        Number(n) => Quoted(format!("gid://gitlab/Iterations::Cadence/{n}")),
        List(arr) => List(arr.iter().map(to_iteration_cadence_id).collect()),
        _ => v.clone(),
    }
}

pub fn to_type_param(s: &str) -> String {
    match s.to_uppercase().as_str() {
        "TESTCASE" => "TEST_CASE",
        "KEYRESULT" => "KEY_RESULT",
        "MERGEREQUEST" => "MERGE_REQUEST",
        _ => s,
    }
    .to_string()
}

fn to_graphql_filter_value(expr: &Expression, context: &mut Context) -> Value {
    match (&expr.field.dealias(), &expr.value) {
        (Id, Number(n)) => Quoted(n.to_string()),
        (Id, List(n)) => List(n.iter().map(|i| Quoted(i.to_string())).collect()),
        (Assignee, Number(n)) => Quoted(n.to_string()),
        (Label | Approver, Token(s)) => Quoted(s.clone()),
        (Label | Assignee | Approver, Null) => Quoted("NONE".to_string()),
        (Weight | Epic, Null) => Token("NONE".to_string()),
        (Weight, Number(n)) => Quoted(n.to_string()),
        (State, Token(s)) => Token(s.to_lowercase()),
        (Status, Quoted(s)) => Token(format!("{{name: \"{s}\"}}")),
        (Type, Token(s)) => Token(to_type_param(s.as_str())),
        (Type, List(types)) => List(
            types
                .iter()
                .map(|i| Token(to_type_param(&i.to_string())))
                .collect(),
        ),
        (Health, v) => to_health_status(v),
        (Cadence, v) => to_iteration_cadence_id(v),
        (Subscribed, Bool(b)) => match b {
            true => Token("EXPLICITLY_SUBSCRIBED".to_string()),
            false => Token("EXPLICITLY_UNSUBSCRIBED".to_string()),
        },
        (Epic, v) => {
            if let Some(mapping) = get_field_mapping(&Epic, context) {
                use crate::field_mapping::{
                    transform_to_legacy_epic_subquery, transform_to_work_item_epic_subquery,
                };
                let transformer = match mapping.variable_type.as_str() {
                    "WorkItemID!" => transform_to_work_item_epic_subquery,
                    "String" => transform_to_legacy_epic_subquery,
                    _ => transform_to_legacy_epic_subquery, // fallback
                };
                let subquery = transformer(v, context);

                // graphql variables are not automatically coerced to lists,
                // so we need to wrap the subquery in a list, if it isn't already
                if FeatureFlag::GlqlWorkItems.get() && matches!(subquery, Subquery(..)) {
                    List(vec![subquery])
                } else {
                    subquery
                }
            } else {
                v.clone()
            }
        }
        (_, v) => v.clone(),
    }
}

lazy_static! {
    static ref HEALTH_STATUSES: IndexMap<String, String> = {
        let mut m = IndexMap::new();
        m.insert("on track".to_string(), "ON_TRACK".to_string());
        m.insert("needs attention".to_string(), "NEEDS_ATTENTION".to_string());
        m.insert("at risk".to_string(), "AT_RISK".to_string());
        m
    };
}

fn to_graphql_sort(sort: &Sort) -> String {
    match &sort.field.dealias() {
        // MergeRequests
        Merged => "merged_at",
        Closed => "closed_at",
        Title => "title",
        Popularity => "popularity",
        Milestone => "milestone_due",
        Updated => "updated",
        Created => "created",

        // Epics
        Start => "start_date",
        Due => "due_date",

        // Issues
        Weight => "weight",
        Health => "health_status",

        // Default
        _ => {
            return to_graphql_sort(&Sort {
                field: Created,
                order: sort.order.clone(),
            });
        }
    }
    .to_uppercase()
    .to_string()
        + "_"
        + &sort.order.to_string().to_uppercase()
}

fn to_graphql_filter_key(expr: &Expression, context: &Context) -> String {
    match (&expr.field.dealias(), &expr.operator, &expr.value) {
        (Assignee, _, Null | Number(_)) => "assigneeId".to_string(),
        (Assignee, _, Token(_)) => "assigneeWildcardId".to_string(),
        (Assignee, _, List(arr)) if arr.is_empty() => "assigneeWildcardId".to_string(),
        (Assignee, Equal, _) if context.source == Some(MergeRequests) => {
            "assigneeUsername".to_string()
        }
        (Assignee, _, _) => pluralize_attribute("assigneeUsername").to_string(),
        (Author, In, _) => pluralize_attribute("authorUsername").to_string(),
        (Author, _, _) => "authorUsername".to_string(),
        (Id, ..) => "iids".to_string(),
        (Type, ..) => "types".to_string(),
        (Approver, ..) => "approvedBy".to_string(),
        (Merger, ..) => "mergedBy".to_string(),
        (Reviewer, _, Token(_)) => "reviewerWildcardId".to_string(),
        (Reviewer, ..) => "reviewerUsername".to_string(),
        (Environment, ..) => "environmentName".to_string(),
        (Epic, _, _) => {
            if let Some(mapping) = get_field_mapping(&Epic, context) {
                (mapping.get_attribute)(&mapping.field_type, &expr.value)
            } else {
                // Fallback to legacy behavior
                match (&expr.value, FeatureFlag::GlqlWorkItems.get()) {
                    (Null | Token(_), true) => "parentWildcardId".to_string(),
                    (Null | Token(_), false) => "epicWildcardId".to_string(),
                    (_, true) => "parentIds".to_string(),
                    (_, false) => "epicId".to_string(),
                }
            }
        }
        (Label, In, _) => pluralize_attribute("labelName").to_string(),
        (Label, _, _) => "labelName".to_string(),
        (Health, _, _) => "healthStatusFilter".to_string(),
        (IncludeSubgroups, _, _) => match FeatureFlag::GlqlWorkItems.get() {
            // work items only
            true if context.source == Some(Issues) => "includeDescendants",
            // legacy epics
            false if context.source == Some(Epics) => "includeDescendantGroups",
            // legacy issues and merge requests
            _ => "includeSubgroups",
        }
        .to_string(),
        (Iteration, _, Token(_)) => "iterationWildcardId".to_string(),
        (Iteration, _, _) => "iterationId".to_string(),
        (Cadence, _, _) => "iterationCadenceId".to_string(),
        (Milestone, _, Token(_)) => "milestoneWildcardId".to_string(),
        (Milestone, _, List(arr)) if arr.is_empty() => "milestoneWildcardId".to_string(),
        (Milestone, _, _) => "milestoneTitle".to_string(),
        (SourceBranch, _, _) => "sourceBranches".to_string(),
        (TargetBranch, _, _) => "targetBranches".to_string(),
        (MyReaction, _, _) => "myReactionEmoji".to_string(),
        (Project, _, _) => {
            panic!("project interpreted as token instead of captured by optimizer")
        }
        (Group, _, _) => panic!("group interpreted as token instead of captured by optimizer"),
        (Weight, _, Null | Token(_)) => "weightWildcardId".to_string(),
        (_, GreaterThan | GreaterThanEquals, _) => format!("{}After", expr.field),
        (_, LessThan | LessThanEquals, _) => format!("{}Before", expr.field),
        (UnknownField(f), _, _) => f.clone(),
        (_, _, _) => expr.field.to_string(),
    }
    .to_string()
}

fn pluralize_attribute(a: &str) -> &str {
    match a {
        "labelName" => "labelNames",
        "assigneeUsername" => "assigneeUsernames",
        "authorUsername" => "authorUsernames",
        _ => a,
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_stringify_multiple_filters() {
        let mut filters = GraphQLFilters::default();

        filters
            .and
            .insert("field1".to_string(), Quoted("value1".to_string()));
        filters.and.insert("field2".to_string(), Number(42));

        filters.or.insert("field3".to_string(), Bool(true));
        filters
            .or
            .insert("field4".to_string(), Quoted("value4".to_string()));

        filters.not.insert(
            "field6".to_string(), // This field will be ignored because not filters are not wrapped in an 'or' clause
            Quoted("value6".to_string()),
        );

        filters.constraints.insert("limit".to_string(), Number(10));

        // Assert exact match
        assert_eq!(
            format!("{filters}"),
            "(field1: \"value1\" field2: 42 or: { field3: true field4: \"value4\" } not: { field6: \"value6\" } limit: 10 )"
        );
    }

    #[test]
    #[should_panic(expected = "project interpreted as token instead of captured by optimizer")]
    fn test_to_graphql_attribute_project_panic() {
        let expr = Expression::new(Project, Equal, Quoted("some_project".into()));
        to_graphql_filter_key(&expr, &Context::default());
    }

    #[test]
    #[should_panic(expected = "group interpreted as token instead of captured by optimizer")]
    fn test_to_graphql_attribute_group_panic() {
        let expr = Expression::new(Group, Equal, Quoted("some_group".into()));
        to_graphql_filter_key(&expr, &Context::default());
    }
}
