use crate::codegen::graphql_filters::GraphQLFilters;
use crate::types::Value::*;
use crate::types::aggregation::DimensionSegment::*;
use crate::types::{
    CompileOutput, Context, DisplayField, DisplayField::*, Query, Source, Source::*,
};
use crate::utils::date_time::TimeRange;
use crate::utils::feature_flags::FeatureFlag;
use graphql_parser::parse_query;

const REQUIRED_FIELDS: &[&str] = &["id", "iid", "title", "webUrl", "reference", "state"];

pub fn generate_code(query: Query, context: &mut Context) -> CompileOutput {
    let filters = GraphQLFilters::new(&query, context);
    let source_name = match context.source {
        Some(Epics) => "epics",
        Some(MergeRequests) => "mergeRequests",
        Some(Issues) | None => match FeatureFlag::GlqlWorkItems.get() {
            true => "workItems",
            false => "issues",
        },
    };

    let required_fields = if !context.is_subquery {
        REQUIRED_FIELDS
            .iter()
            .map(|field| render_field(field))
            .collect::<String>()
    } else {
        "".to_string()
    };

    let mut output_fields = context.fields.clone();
    let unscoped_fragment = if let Some(aggregate) = &context.aggregate
        && let aggregated_fields = render_fields(&aggregate.metrics, &context.source)
        && let Some(dimension) = aggregate.dimensions.first()
        && let field_to_aggregate_on = match &dimension.field.dealias() {
            Static(field) => field.dealias().to_string(),
            _ => "".to_string(),
        }
        && let before_key = &format!("{field_to_aggregate_on}Before")
        && let after_key = &format!("{field_to_aggregate_on}After")
        && let Some(Date(before_value)) = filters.and.get(before_key)
        && let Some(Date(after_value)) = filters.and.get(after_key)
        && let TimeSegment(quantity, unit, segment_type) = &dimension.segment
        && let Ok(date_range) = TimeRange::from_str(after_value, before_value)
        && let Ok(segments) = date_range.segment(*quantity, unit.clone(), segment_type.clone())
        && let aggregated_fragment = segments
            .iter()
            .map(|segment| {
                let mut segment_filters = filters.clone();
                segment_filters.and.insert(
                    before_key.to_string(),
                    Quoted(segment.to.format("%Y-%m-%d %H:%M").to_string()),
                );
                segment_filters.and.insert(
                    after_key.to_string(),
                    Quoted(segment.from.format("%Y-%m-%d %H:%M").to_string()),
                );
                segment_filters
                    .constraints
                    .insert("first".to_string(), Number(0));
                segment_filters.constraints.shift_remove("before");
                segment_filters.constraints.shift_remove("after");

                format!(
                    "{segment_name}: {source_name}{segment_filters} {{{aggregated_fields}}} ",
                    segment_name = segment.key(),
                )
            })
            .collect::<String>()
        && !aggregated_fragment.is_empty()
    {
        output_fields = aggregate
            .dimensions
            .iter()
            .map(|d| d.field.clone())
            .chain(aggregate.metrics.clone())
            .collect();

        context.variables = context
            .variables
            .iter()
            .filter_map(|v| match v.key.as_str() {
                "before" | "after" | "limit" => None,
                _ => Some(v.to_owned()),
            })
            .collect::<Vec<_>>();

        aggregated_fragment
    } else {
        format!(
            "{source_name}{filters} {{ nodes {{{required_fields} {fields}}} {page_info}}}",
            fields = render_fields(&context.fields, &context.source),
            page_info = match context.include_page_info() {
                true => "pageInfo { startCursor endCursor hasNextPage hasPreviousPage } count",
                false => "",
            }
        )
    };

    let output = format!(
        "query GLQL{variables} {{{scoped_fragment}}}",
        variables = match &context.variables {
            arr if arr.is_empty() => "".to_string(),
            arr => format!(
                "({})",
                arr.iter().map(|v| v.to_string()).collect::<String>()
            ),
        },
        scoped_fragment = if let Some(project_name) = &context.project {
            format!("project(fullPath: \"{project_name}\") {{{unscoped_fragment}}}")
        } else if let Some(group_name) = &context.group {
            format!("group(fullPath: \"{group_name}\") {{{unscoped_fragment}}}")
        } else {
            unscoped_fragment.to_string()
        }
    );

    let output = format!("{parsed}", parsed = parse_query::<String>(&output).unwrap());

    CompileOutput::success(output)
        .with_variables(context.variables.clone())
        .with_fields(output_fields)
}

fn render_fields(fields: &[DisplayField], source: &Option<Source>) -> String {
    fields
        .iter()
        .map(|field| render_work_item_widget(&field.name(), source))
        .collect::<String>()
}

fn render_work_item_widget(field: &str, source: &Option<Source>) -> String {
    if !FeatureFlag::GlqlWorkItems.get() || source != &Some(Issues) {
        return render_field(field);
    }

    let mut field = field;
    let widget_name = match field {
        "assignees" => "WorkItemWidgetAssignees",
        "labels" => "WorkItemWidgetLabels",
        "milestone" => "WorkItemWidgetMilestone",
        "startDate" | "dueDate" => "WorkItemWidgetStartAndDueDate",
        "timeEstimate" | "totalTimeSpent" => "WorkItemWidgetTimeTracking",
        "healthStatus" => "WorkItemWidgetHealthStatus",
        "iteration" => "WorkItemWidgetIteration",
        "weight" => "WorkItemWidgetWeight",
        "progress" => "WorkItemWidgetProgress",
        "color" => "WorkItemWidgetColor",
        "epic" | "parent" => {
            field = "parent";
            "WorkItemWidgetHierarchy"
        }
        "taskCompletionStatus" => "WorkItemWidgetDescription",
        "lastComment" => "WorkItemWidgetNotes",
        "status" => "WorkItemWidgetStatus",
        _ => return render_field(field),
    };

    format!(
        "widgets {{ ... on {widget_name} {{ type {field} }} }} ",
        field = render_field(field)
    )
}

fn render_field(field: &str) -> String {
    match field {
        "labels" => format!("{field} {{ nodes {{ id title color textColor }} }}"),
        "assignees" | "approvedBy" | "reviewers" => {
            format!("{field} {{ nodes {{ id avatarUrl username name webUrl }} }}")
        }
        "author" | "mergeUser" => {
            format!("{field} {{ id avatarUrl username name webUrl }}")
        }
        "milestone" => format!("{field} {{ id iid dueDate startDate title webPath }}"),
        "epic" | "parent" => format!("{field} {{ id iid reference state title webUrl }}"),
        "color" => "color textColor".to_string(),
        "progress" => "currentValue endValue progress startValue updatedAt".to_string(),
        "iteration" => format!(
            "{field} {{ id iid startDate dueDate title webUrl iterationCadence {{ id title }} }}"
        ),
        "taskCompletionStatus" => format!("{field} {{ completedCount count }}"),
        "lastComment" => {
            format!("{field}:notes(filter: ONLY_COMMENTS last: 1) {{ nodes {{ bodyHtml }} }}")
        }
        "project" | "sourceProject" | "targetProject" => {
            format!("{field} {{ fullPath webUrl nameWithNamespace }}")
        }
        "group" => format!("{field} {{ fullPath webUrl fullName }}"),
        "workItemType" if FeatureFlag::GlqlWorkItems.get() => {
            "workItemType { name iconName }".to_string()
        }
        "status" => format!("{field} {{ category color description iconName name }}"),
        _ => format!("{field}\n"),
    }
}
