use crate::types::{
    Field,
    FieldType::{self, *},
    Source,
    Value::{self, *},
};

use crate::errors::{GlqlError, GlqlErrorKind};

pub struct ValueAnalyzer {
    value: Value,
    source: Source,
}

impl ValueAnalyzer {
    pub fn new(value: Value, source: Source) -> Self {
        Self { value, source }
    }

    pub fn analyze(&self, field: &Field) -> Result<(), GlqlError> {
        const MAX_ARRAY_SIZE: usize = 100;

        match &self.value {
            List(items) if items.len() > MAX_ARRAY_SIZE => {
                Err(GlqlError::analyzer_error(GlqlErrorKind::ArrayTooLarge {
                    field: field.clone(),
                    max_allowed: MAX_ARRAY_SIZE,
                }))
            }
            _ => Ok(()),
        }
    }

    pub fn field_type(&self, field: &Field) -> Option<FieldType> {
        let field_type = self.source.analyzer().field_type(field);
        let all_field_types = self.source.analyzer().all_field_types(field);

        match &self.value {
            List(l) if field_type.is_list_like() && !l.is_empty() => {
                field_type.find(|f| matches!(f, ListLike(..)))
            }
            Token(t)
                if field_type.is_enum_like()
                    && field_type.enum_values().contains(&t.to_lowercase()) =>
            {
                field_type.find(|f| matches!(f, EnumLike(_) | StringEnumLike(_)))
            }
            _ => all_field_types.iter().fold(None, |acc, type_| {
                if type_.analyzer().analyze(&self.value) {
                    Some(type_.clone())
                } else {
                    acc
                }
            }),
        }
    }
}
