use crate::errors::{GlqlError, GlqlErrorKind::*};
use crate::types::{
    Field,
    FieldType::{self, *},
    Sort,
    Source::{self, *},
};
use epics::EpicsSourceAnalyzer;
use issues::IssuesSourceAnalyzer;
use merge_requests::MergeRequestsSourceAnalyzer;

mod epics;
mod issues;
mod merge_requests;
mod shared;

pub trait SourceAnalyzer {
    fn source(&self) -> Source;
    fn is_valid_field(&self, field: &Field) -> bool;
    fn field_type(&self, field: &Field) -> FieldType;
    fn valid_sort_fields(&self) -> Vec<Field>;

    fn all_field_types(&self, field: &Field) -> Vec<FieldType> {
        let field_type = self.field_type(field);
        if let Multiple(fields) = field_type {
            *fields
        } else {
            vec![self.field_type(field)]
        }
    }

    fn analyze_field(&self, field: &Field) -> Result<(), GlqlError> {
        if self.is_valid_field(field) {
            Ok(())
        } else {
            Err(GlqlError::analyzer_error(UnrecognizedFieldForSource {
                field: field.clone(),
                source: self.source().clone(),
            }))
        }
    }

    fn analyze_sort_field(&self, sort: &Sort) -> Result<(), GlqlError> {
        let valid_sort_fields = self.valid_sort_fields();
        if !valid_sort_fields.contains(sort.field.dealias()) {
            return Err(GlqlError::analyzer_error(UnrecognizedSortFieldForSource {
                field: sort.field.clone(),
                source: self.source().clone(),
                supported_fields: valid_sort_fields.clone(),
            }));
        }

        Ok(())
    }
}

impl Source {
    pub fn analyzer(&self) -> &dyn SourceAnalyzer {
        match &self {
            Epics => &EpicsSourceAnalyzer,
            Issues => &IssuesSourceAnalyzer,
            MergeRequests => &MergeRequestsSourceAnalyzer,
        }
    }
}
