use super::shared::type_field_type;
use crate::{
    analyzer::sources::SourceAnalyzer,
    types::{
        Field::{self, *},
        FieldType::{self, *},
        Operator::*,
        ReferenceType::*,
        RelationshipType::*,
        Source::{self, *},
    },
};

pub struct MergeRequestsSourceAnalyzer;

impl SourceAnalyzer for MergeRequestsSourceAnalyzer {
    fn source(&self) -> Source {
        MergeRequests
    }

    fn is_valid_field(&self, field: &Field) -> bool {
        matches!(
            field.dealias(),
            Type | Id
                | Label
                | Created
                | Updated
                | Assignee
                | Author
                | Milestone
                | Group
                | Project
                | Reviewer
                | Approver
                | Merger
                | State
                | Merged
                | Deployed
                | Draft
                | Environment
                | IncludeSubgroups
                | SourceBranch
                | TargetBranch
                | MyReaction
                | Subscribed
        )
    }

    fn valid_sort_fields(&self) -> Vec<Field> {
        vec![
            Merged, Closed, Title, Popularity, Milestone, Updated, Created,
        ]
    }

    fn field_type(&self, field: &Field) -> FieldType {
        match field.dealias() {
            Type => type_field_type(),

            Id => {
                NumberLike.with_ops([Equal]) | ListLike(HasOne, Box::new(NumberLike)).with_ops([In])
            }
            State => EnumLike(vec![
                "opened".into(),
                "closed".into(),
                "merged".into(),
                "all".into(),
            ])
            .with_ops([Equal]),

            Assignee | Reviewer => StringLike | ReferenceLike(UserRef) | Nullable,
            Approver => {
                StringLike
                    | ReferenceLike(UserRef)
                    | Nullable
                    | ListLike(HasMany, Box::new(StringLike | ReferenceLike(UserRef)))
                        .with_ops([Equal, NotEqual])
            }
            Author => StringLike | ReferenceLike(UserRef),
            Merger => StringLike.with_ops([Equal]) | ReferenceLike(UserRef).with_ops([Equal]),
            Label => {
                StringLike
                    | ListLike(HasMany, Box::new(StringLike | ReferenceLike(LabelRef)))
                        .with_ops([Equal, NotEqual])
                    | Nullable
                    | ReferenceLike(LabelRef)
            }
            Milestone => {
                StringLike
                    | Nullable
                    | EnumLike(vec!["upcoming".into(), "started".into()])
                    | ReferenceLike(MilestoneRef)
            }

            SourceBranch | TargetBranch => StringLike | ListLike(HasOne, Box::new(StringLike)),
            Environment | Group | Project => StringLike.with_ops([Equal]),
            Created | Updated | Merged | Deployed => DateLike,
            Draft => BooleanLike,
            IncludeSubgroups => BooleanLike.paired_with([Group]),
            MyReaction => StringLike,
            Subscribed => BooleanLike,
            _ => UnknownFieldType,
        }
    }
}
