use super::shared::type_field_type;
use crate::analyzer::sources::SourceAnalyzer;
use crate::field_mapping::get_field_mapping;
use crate::types::{
    Context,
    Field::{self, *},
    FieldType::{self, *},
    Operator::*,
    ReferenceType::*,
    RelationshipType::*,
    Source::{self, *},
};
use crate::utils::feature_flags::FeatureFlag;
pub struct IssuesSourceAnalyzer;

impl SourceAnalyzer for IssuesSourceAnalyzer {
    fn source(&self) -> Source {
        Issues
    }

    fn is_valid_field(&self, field: &Field) -> bool {
        matches!(
            field.dealias(),
            Type | Id
                | State
                | Status
                | Label
                | Epic
                | Created
                | Updated
                | Assignee
                | Author
                | Milestone
                | Weight
                | Group
                | Project
                | Confidential
                | Health
                | Closed
                | Iteration
                | Cadence
                | Due
                | IncludeSubgroups
                | MyReaction
                | Subscribed
                | CustomField(_)
        )
    }

    fn valid_sort_fields(&self) -> Vec<Field> {
        let mut valid_fields = vec![
            Due, Title, Popularity, Closed, Weight, Health, Milestone, Updated, Created,
        ];
        if FeatureFlag::GlqlWorkItems.get() {
            valid_fields.push(Start);
        }
        valid_fields
    }

    fn field_type(&self, field: &Field) -> FieldType {
        match field.dealias() {
            Type => type_field_type(),

            Id => {
                NumberLike.with_ops([Equal]) | ListLike(HasOne, Box::new(NumberLike)).with_ops([In])
            }
            State => {
                EnumLike(vec!["opened".into(), "closed".into(), "all".into()]).with_ops([Equal])
            }
            Status => StringLike.with_ops([Equal]),

            Assignee => {
                StringLike
                    | ReferenceLike(UserRef)
                    | ListLike(HasMany, Box::new(StringLike | ReferenceLike(UserRef)))
                    | NumberLike
                    | Nullable
            }
            Author => {
                StringLike
                    | ReferenceLike(UserRef)
                    // hack: Technically, `author` is a HasOne field, but the backend supports it
                    // differently from other HasOne fields like milestone and iteration.
                    | ListLike(HasMany, Box::new(StringLike | ReferenceLike(UserRef)))
                        .with_ops([In, NotEqual])
            }
            Label => {
                StringLike
                    | ListLike(HasMany, Box::new(StringLike | ReferenceLike(LabelRef)))
                    | Nullable
                    | ReferenceLike(LabelRef)
            }
            Cadence => {
                StringLike.with_ops([Equal])
                    | ListLike(HasOne, Box::new(NumberLike | StringLike)).with_ops([In])
                    | NumberLike.with_ops([Equal])
                    | Nullable
            }
            Milestone => {
                StringLike
                    | ListLike(HasOne, Box::new(StringLike | ReferenceLike(MilestoneRef)))
                    | Nullable
                    | EnumLike(vec!["upcoming".into(), "started".into()])
                    | ReferenceLike(MilestoneRef)
            }
            Iteration => {
                StringLike
                    | ListLike(
                        HasOne,
                        Box::new(NumberLike | StringLike | ReferenceLike(IterationRef)),
                    )
                    | NumberLike
                    | Nullable
                    | EnumLike(vec!["current".into()])
                    | ReferenceLike(IterationRef)
            }
            Epic => {
                // Use the new field mapping abstraction for Epic field
                let context = Context {
                    source: Some(crate::types::Source::Issues),
                    ..Context::default()
                };
                if let Some(mapping) = get_field_mapping(field, &context) {
                    mapping.field_type
                } else {
                    UnknownFieldType
                }
            }

            Group | Project => StringLike.with_ops([Equal]),
            Closed | Created | Updated | Due => DateLike,
            Weight => NumberLike | Nullable,
            Confidential => BooleanLike,
            IncludeSubgroups => BooleanLike.paired_with([Group]),
            Health => {
                StringEnumLike(vec![
                    "on track".into(),
                    "needs attention".into(),
                    "at risk".into(),
                ]) | Nullable
            }
            MyReaction => StringLike,
            Subscribed => BooleanLike,

            CustomField(_) => {
                StringLike.with_ops([Equal])
                    | ListLike(HasMany, Box::new(StringLike)).with_ops([Equal])
            }
            _ => UnknownFieldType,
        }
    }
}
