use super::shared::type_field_type;
use crate::{
    analyzer::sources::SourceAnalyzer,
    types::{
        Field::{self, *},
        FieldType::{self, *},
        Operator::*,
        ReferenceType::*,
        RelationshipType::*,
        Source::{self, *},
    },
};

pub struct EpicsSourceAnalyzer;

impl SourceAnalyzer for EpicsSourceAnalyzer {
    fn source(&self) -> Source {
        Epics
    }

    fn is_valid_field(&self, field: &Field) -> bool {
        matches!(
            field.dealias(),
            Type | Id
                | State
                | Author
                | Label
                | Milestone
                | Group
                | Created
                | Updated
                | Confidential
                | IncludeSubgroups
        )
    }

    fn valid_sort_fields(&self) -> Vec<Field> {
        vec![Start, Due, Title, Created, Updated]
    }

    fn field_type(&self, field: &Field) -> FieldType {
        match field.dealias() {
            Type => type_field_type(),

            Id => {
                NumberLike.with_ops([Equal]) | ListLike(HasOne, Box::new(NumberLike)).with_ops([In])
            }
            State => {
                EnumLike(vec!["opened".into(), "closed".into(), "all".into()]).with_ops([Equal])
            }
            Author => {
                StringLike
                    | ReferenceLike(UserRef)
                    // hack: Technically, `author` is a HasOne field, but the backend supports it
                    // differently from other HasOne fields like milestone and iteration.
                    | ListLike(HasMany, Box::new(StringLike | ReferenceLike(UserRef)))
                        .with_ops([In])
            }
            Label => {
                StringLike
                    | ListLike(HasMany, Box::new(StringLike | ReferenceLike(LabelRef)))
                    | Nullable
                    | ReferenceLike(LabelRef)
            }
            Milestone => {
                StringLike
                    | ListLike(HasOne, Box::new(StringLike | ReferenceLike(MilestoneRef)))
                    | Nullable
                    | EnumLike(vec!["upcoming".into(), "started".into()])
                    | ReferenceLike(MilestoneRef)
            }
            Group => StringLike.with_ops([Equal]),
            Created | Updated => DateLike,
            Confidential => BooleanLike,
            IncludeSubgroups => BooleanLike.paired_with([Group]),
            _ => UnknownFieldType,
        }
    }
}
