use crate::errors::GlqlErrorKind::*;
use crate::types::{SortOrder::*, Source};
use crate::{errors::GlqlError, types::Sort};

pub struct SortAnalyzer {
    sort: Sort,
    source: Source,
}

impl SortAnalyzer {
    pub fn new(sort: Sort, source: Source) -> Self {
        Self { sort, source }
    }

    pub fn analyze(&self) -> Result<(), GlqlError> {
        self.source.analyzer().analyze_sort_field(&self.sort)?;

        Ok(())
    }

    pub fn analyze_order(&self) -> Result<(), GlqlError> {
        if let UnknownSortOrder(s) = &self.sort.order {
            return Err(GlqlError::analyzer_error(InvalidSortOrder {
                order: s.clone(),
            }));
        }

        Ok(())
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::types::Field::*;

    #[test]
    fn test_analyze_order_with_valid_orders() {
        let sort = Sort {
            field: Title,
            order: Ascending,
        };
        assert!(
            SortAnalyzer::new(sort, Source::Issues)
                .analyze_order()
                .is_ok()
        );

        let sort = Sort {
            field: Title,
            order: Descending,
        };
        assert!(
            SortAnalyzer::new(sort, Source::Issues)
                .analyze_order()
                .is_ok()
        );
    }

    #[test]
    fn test_analyze_order_with_invalid_order() {
        let sort = Sort {
            field: Title,
            order: UnknownSortOrder("invalid".to_string()),
        };
        let result = SortAnalyzer::new(sort, Source::Issues).analyze_order();
        assert!(result.is_err());
        assert_eq!(
            result.unwrap_err().to_string(),
            "Error: Invalid sort order: `invalid`. Valid sort orders: `asc`, `desc`."
        );
    }
}
