use crate::analyzer::expression::ExpressionAnalyzer;
use crate::analyzer::field::FieldAnalyzer;
use crate::errors::{GlqlError, GlqlErrorKind::*};
use crate::types::{Operator::*, Query, Source};
use indexmap::set::IndexSet;

pub struct QueryAnalyzer {
    query: Query,
    source: Source,
}

impl QueryAnalyzer {
    pub fn new(query: Query, source: Source) -> Self {
        Self { query, source }
    }

    pub fn analyze(&self) -> Result<(), GlqlError> {
        self.analyze_expressions()
    }

    fn analyze_expressions(&self) -> Result<(), GlqlError> {
        let mut in_operators = IndexSet::new();
        let fields = self
            .query
            .expressions
            .iter()
            .map(|expr| expr.field.clone())
            .collect::<Vec<_>>();

        for expr in &self.query.expressions {
            ExpressionAnalyzer::new(expr.clone(), self.source.clone()).analyze()?;
            FieldAnalyzer::new(expr.field.clone(), self.source.clone())
                .analyze_paired_fields(fields.clone())?;

            if expr.operator == In && !in_operators.insert(expr.field.clone()) {
                Err(GlqlError::analyzer_error(DuplicateOperatorForField {
                    field: expr.field.clone(),
                }))?;
            }
        }

        Ok(())
    }
}
