use crate::analyzer::context::ContextAnalyzer;
use crate::analyzer::query::QueryAnalyzer;
use crate::errors::{GlqlError, GlqlErrorKind::*};
use crate::types::{Context, Query, Source};

pub mod context;
pub mod expression;
pub mod field;
pub mod field_type;
pub mod query;
pub mod sort;
pub mod sources;
pub mod value;

pub fn analyze(query: &Query, context: &Context) -> Result<(), GlqlError> {
    let source = context_source(context)?;
    QueryAnalyzer::new(query.clone(), source.clone()).analyze()?;
    ContextAnalyzer::new(context.clone(), source.clone()).analyze()?;

    Ok(())
}

fn context_source(context: &Context) -> Result<&Source, GlqlError> {
    context
        .source
        .as_ref()
        .ok_or(GlqlError::analyzer_error(SourceNotProvided))
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::types::{Expression, Field::*, Operator::*, Value::*};

    #[test]
    fn test_analyze_invalid_field() {
        let query = Query::new(vec![Expression::new(
            UnknownField("foo".into()),
            Equal,
            Quoted("foo".into()),
        )]);
        assert_eq!(
            analyze(&query, &Context::default())
                .unwrap_err()
                .to_string(),
            "Error: foo is not a recognized field."
        );
    }

    #[test]
    fn test_invalid_weight_operator() {
        let query = Query::new(vec![Expression::new(Weight, GreaterThan, Number(1))]);
        assert_eq!(
            analyze(&query, &Context::default())
                .unwrap_err()
                .to_string(),
            "Error: `weight` does not support the greater than (`>`) operator. Supported operators: equals (`=`), not equals (`!=`)."
        );

        let query2 = Query::new(vec![Expression::new(Weight, LessThan, Number(1))]);
        assert_eq!(
            analyze(&query2, &Context::default())
                .unwrap_err()
                .to_string(),
            "Error: `weight` does not support the less than (`<`) operator. Supported operators: equals (`=`), not equals (`!=`)."
        );
    }

    #[test]
    fn test_invalid_in_with_non_array() {
        let query = Query::new(vec![Expression::new(Label, In, Quoted("fish".into()))]);
        assert_eq!(
            analyze(&query, &Context::default())
                .unwrap_err()
                .to_string(),
            "Error: `label` does not support the is one of (`in`) operator for `\"fish\"`. Supported operators: equals (`=`), not equals (`!=`)."
        );
    }

    #[test]
    fn test_invalid_in_with_empty_array() {
        let query = Query::new(vec![Expression::new(Label, In, List(vec![]))]);
        assert_eq!(
            analyze(&query, &Context::default())
                .unwrap_err()
                .to_string(),
            "Error: `label` does not support the is one of (`in`) operator for `()`. Supported operators: equals (`=`), not equals (`!=`)."
        );
    }

    #[test]
    fn test_invalid_use_of_list() {
        let query = Query::new(vec![Expression::new(
            Created,
            Equal,
            List(vec![Quoted("fish".into())]),
        )]);
        assert_eq!(
            analyze(&query, &Context::default())
                .unwrap_err()
                .to_string(),
            "Error: `created` cannot be compared with `(\"fish\")`. Supported value types: `Date`."
        );
    }

    #[test]
    fn test_invalid_health_status() {
        let query = Query::new(vec![Expression::new(Health, Equal, Quoted("foo".into()))]);
        assert_eq!(
            analyze(&query, &Context::default())
                .unwrap_err()
                .to_string(),
            "Error: `health` cannot be compared with `\"foo\"`. Supported value types: `StringEnum` (`\"on track\"`, `\"needs attention\"`, `\"at risk\"`), `Nullable` (`null`, `none`, `any`)."
        );
    }

    #[test]
    fn test_multiple_in_for_same_field() {
        let query = Query::new(vec![
            Expression::new(Label, In, List(vec![Quoted("foo".into())])),
            Expression::new(Label, In, List(vec![Quoted("bar".into())])),
        ]);
        assert_eq!(
            analyze(&query, &Context::default())
                .unwrap_err()
                .to_string(),
            "Error: The is one of (`in`) operator can only be used once with `label`."
        );
    }

    #[test]
    fn test_opened_closed_field_validation() {
        // Test invalid string value
        let query = Query::new(vec![Expression::new(
            Created.aliased_as("opened"),
            Equal,
            Quoted("fish".into()),
        )]);
        assert_eq!(
            analyze(&query, &Context::default())
                .unwrap_err()
                .to_string(),
            "Error: `opened` cannot be compared with `\"fish\"`. Supported value types: `Date`."
        );

        // Test invalid number value
        let query = Query::new(vec![Expression::new(Closed, Equal, Number(2))]);
        assert_eq!(
            analyze(&query, &Context::default())
                .unwrap_err()
                .to_string(),
            "Error: `closed` cannot be compared with `2`. Supported value types: `Date`."
        );
    }

    #[test]
    fn test_invalid_milestone_iteration_cadence_equals_with_list() {
        let test_cases = vec![
            (
                "milestone",
                "Error: `milestone` does not support the equals (`=`) operator for `(\"milestone 1\")`. Supported operators: is one of (`in`), not equals (`!=`).",
            ),
            (
                "iteration",
                "Error: `iteration` does not support the equals (`=`) operator for `(\"iteration 1\")`. Supported operators: is one of (`in`), not equals (`!=`).",
            ),
            (
                "cadence",
                "Error: `cadence` does not support the equals (`=`) operator for `(\"cadence 1\")`. Supported operators: is one of (`in`).",
            ),
        ];

        for (field, expected_error) in test_cases {
            let query = Query::new(vec![Expression::new(
                field.into(),
                Equal,
                List(vec![Quoted(format!("{field} 1"))]),
            )]);
            assert_eq!(
                analyze(&query, &Context::default())
                    .unwrap_err()
                    .to_string(),
                expected_error
            );
        }
    }

    #[test]
    fn test_milestone_iteration_cadence_equal_with_array() {
        let test_cases = vec![
            ("milestone", ("17.5", "Backlog")),
            ("iteration", ("12", "34")),
            ("cadence", ("12", "34")),
        ];

        for (field, value) in test_cases {
            let query = Query::new(vec![Expression::new(
                field.into(),
                Equal,
                List(vec![
                    Quoted(value.0.to_string()),
                    Quoted(value.1.to_string()),
                ]),
            )]);
            assert!(analyze(&query, &Context::default()).is_err());
        }
    }

    #[test]
    fn test_valid_milestone_iteration_cadence_in_operator() {
        let test_cases = vec![
            ("milestone", "Milestone 1"),
            ("iteration", "Iteration 1"),
            ("cadence", "Cadence 1"),
        ];

        for (field, value) in test_cases {
            let query = Query::new(vec![Expression::new(
                field.into(),
                In,
                List(vec![Quoted(value.to_string())]),
            )]);

            assert!(
                analyze(&query, &Context::default()).is_ok(),
                "Validation failed for field: {field} with value: {value}"
            );
        }
    }

    #[test]
    fn test_array_size_limit() {
        let valid_array = (1..=100).map(|i| Quoted(format!("item{i}"))).collect();
        let query = Query::new(vec![Expression::new(Label, In, List(valid_array))]);

        assert!(
            analyze(&query, &Context::default()).is_ok(),
            "Query with 100 items should pass validation"
        );

        let large_array = (1..=101).map(|i| Quoted(format!("item{i}"))).collect();
        let query = Query::new(vec![Expression::new(Label, In, List(large_array))]);

        assert_eq!(
            analyze(&query, &Context::default())
                .unwrap_err()
                .to_string(),
            "Error: `label` field exceeds maximum limit of 100 items."
        );
    }
}
