use crate::types::{
    Field,
    FieldType::{self, *},
    Operator::{self, *},
    RelationshipType::*,
    Value::{self, *},
};
use chrono::NaiveDate;
use indexmap::set::IndexSet;

pub struct FieldTypeAnalyzer {
    field_type: FieldType,
}

impl FieldTypeAnalyzer {
    fn new(field_type: FieldType) -> Self {
        Self { field_type }
    }

    pub fn analyze(&self, value: &Value) -> bool {
        match &self.field_type {
            Nullable => {
                matches!(value, Token(t) if matches!(t.to_lowercase().as_str(), "none" | "any"))
                    || matches!(value, Null)
                    || matches!(value, List(l) if l.is_empty())
            }
            StringLike => matches!(value, Quoted(_)),
            ReferenceLike(r) => {
                matches!(value, Reference(t, _) if t == r)
            }
            NumberLike => matches!(value, Number(_)),
            DateLike => match value {
                Date(s) | Quoted(s) => NaiveDate::parse_from_str(s, "%Y-%m-%d").is_ok(),
                _ => false,
            },
            BooleanLike => match value {
                Null | Bool(_) => true,
                Number(n) if *n == 0 || *n == 1 => true,
                Quoted(s) if matches!(s.to_lowercase().as_str(), "true" | "false" | "0" | "1") => {
                    true
                }
                _ => false,
            },
            ListLike(_, list_type) => {
                matches!(value, List(items) if items.iter().all(|v| list_type.analyzer().analyze(v)))
            }
            EnumLike(values) | StringEnumLike(values) => match value {
                Token(v) | Quoted(v) => values
                    .iter()
                    .map(|v| v.to_lowercase())
                    .collect::<Vec<_>>()
                    .contains(&v.to_lowercase()),
                _ => false,
            },
            WithOperators(field, _) | PairedWith(field, _) => field.analyzer().analyze(value),
            Multiple(fields) => fields.iter().any(|f| f.analyzer().analyze(value)),
            UnknownFieldType => false,
        }
    }

    pub fn pairable_fields(&self) -> Vec<Field> {
        match &self.field_type {
            Multiple(f) => f
                .iter()
                .flat_map(|f| f.analyzer().pairable_fields())
                .collect::<Vec<_>>(),
            WithOperators(f, _) => f.analyzer().pairable_fields(),

            PairedWith(_, f) => f.to_vec(),
            _ => vec![],
        }
    }

    pub fn operators(&self) -> IndexSet<Operator> {
        match &self.field_type {
            Nullable | StringLike | NumberLike | BooleanLike | EnumLike(_) | StringEnumLike(_)
            | ReferenceLike(..) => vec![Equal, NotEqual].into_iter().collect(),
            WithOperators(_, ops) => ops.clone().into_iter().collect(),
            DateLike => vec![
                Equal,
                GreaterThan,
                GreaterThanEquals,
                LessThan,
                LessThanEquals,
            ]
            .into_iter()
            .collect(),
            ListLike(HasOne, _) => vec![In, NotEqual].into_iter().collect(),
            ListLike(HasMany, _) => vec![In, Equal, NotEqual].into_iter().collect(),
            PairedWith(f, _) => f.analyzer().operators(),
            Multiple(_) | UnknownFieldType => vec![].into_iter().collect(),
        }
    }
}

impl FieldType {
    pub fn analyzer(&self) -> FieldTypeAnalyzer {
        FieldTypeAnalyzer::new(self.clone())
    }
}
