use crate::analyzer::value::ValueAnalyzer;
use crate::errors::{GlqlError, GlqlErrorKind::*};
use crate::types::{
    Field::{self, *},
    Operator, Source, Value,
};
use indexmap::set::IndexSet;

pub struct FieldAnalyzer {
    field: Field,
    source: Source,
}

impl FieldAnalyzer {
    pub fn new(field: Field, source: Source) -> Self {
        Self { field, source }
    }

    pub fn operators(&self) -> IndexSet<Operator> {
        self.source
            .analyzer()
            .all_field_types(&self.field)
            .iter()
            .map(|f| f.analyzer().operators())
            .fold(IndexSet::new(), |mut acc, set| {
                acc.extend(set);
                acc
            })
    }

    pub fn operators_for_value(&self, value: &Value) -> IndexSet<Operator> {
        if let Some(field_type) =
            ValueAnalyzer::new(value.clone(), self.source.clone()).field_type(&self.field)
        {
            field_type.analyzer().operators()
        } else {
            IndexSet::new()
        }
    }

    pub fn analyze_field(&self) -> Result<(), GlqlError> {
        match &self.field {
            UnknownField(f) => Err(GlqlError::analyzer_error(UnrecognizedField {
                field: f.to_string(),
            })),
            _ => Ok(()),
        }
    }

    pub fn analyze_field_for_source(&self) -> Result<(), GlqlError> {
        self.source.analyzer().analyze_field(&self.field)
    }

    pub fn analyze_paired_fields(&self, fields: Vec<Field>) -> Result<(), GlqlError> {
        let pairable_fields = self
            .source
            .analyzer()
            .field_type(&self.field)
            .analyzer()
            .pairable_fields();

        match fields
            .iter()
            .any(|f| pairable_fields.contains(f) || pairable_fields.is_empty())
        {
            true => Ok(()),
            false => Err(GlqlError::analyzer_error(IncorrectFieldPairing {
                field: self.field.clone(),
                supported_fields: pairable_fields,
            })),
        }
    }

    pub fn analyze_operator(&self, op: &Operator) -> Result<(), GlqlError> {
        let ops = self.operators();
        let supported_operators = ops.iter().cloned().collect::<Vec<_>>();
        match ops.contains(op) {
            true => Ok(()),
            false => Err(GlqlError::analyzer_error(UnsupportedOperatorForField {
                field: self.field.clone(),
                operator: op.clone(),
                supported_operators,
            })),
        }
    }

    pub fn analyze_operator_for_value(
        &self,
        op: &Operator,
        value: &Value,
    ) -> Result<(), GlqlError> {
        let ops = self.operators_for_value(value);
        let ops_string = ops
            .iter()
            .map(ToString::to_string)
            .collect::<Vec<_>>()
            .join(", ");

        match ops.contains(op) {
            true => Ok(()),
            false => Err(GlqlError::analyzer_error(
                UnsupportedOperatorValueForField {
                    field: self.field.clone(),
                    operator: op.clone(),
                    value: value.clone(),
                    supported_operators: ops_string,
                },
            )),
        }
    }

    pub fn analyze_value(&self, value: &Value) -> Result<(), GlqlError> {
        let types = self.source.analyzer().all_field_types(&self.field);

        match types.iter().find(|t| t.analyzer().analyze(value)) {
            Some(_) => Ok(()),
            None => Err(GlqlError::analyzer_error(UnsupportedValueForField {
                field: self.field.clone(),
                value: value.clone(),
                supported_value_types: types,
            })),
        }
    }

    pub fn analyze(&self, op: &Operator, value: &Value) -> Result<(), GlqlError> {
        self.analyze_field()?;
        self.analyze_field_for_source()?;
        self.analyze_operator(op)?;
        self.analyze_value(value)?;
        self.analyze_operator_for_value(op, value)
    }
}
