#!/bin/sh
set -e

REVIEWERS="himkp drosse"
REMOTE_URL="git@gitlab.com:gitlab-org/glql.git"
RELEASE_BOT_NAME="GLQL Release Bot"

if [ -z "$BUMP_TYPE" ]; then
  BUMP_TYPE="patch"
fi

CURRENT_VERSION=$(grep '^version =' Cargo.toml | sed -E 's/version = "([^"]+)"/\1/')
if [ -z "$CURRENT_VERSION" ]; then
  echo "Could not find version in Cargo.toml"
  exit 1
fi

echo "Current version: $CURRENT_VERSION"

# Split version into components
MAJOR=$(echo "$CURRENT_VERSION" | cut -d'.' -f1)
MINOR=$(echo "$CURRENT_VERSION" | cut -d'.' -f2)
PATCH=$(echo "$CURRENT_VERSION" | cut -d'.' -f3)

# Calculate new version
case $BUMP_TYPE in
  "major")
    MAJOR=$((MAJOR + 1))
    MINOR=0
    PATCH=0
    ;;
  "minor")
    MINOR=$((MINOR + 1))
    PATCH=0
    ;;
  "patch")
    PATCH=$((PATCH + 1))
    ;;
esac

NEW_VERSION="$MAJOR.$MINOR.$PATCH"
echo "New $BUMP_TYPE version: $NEW_VERSION"

echo "Fetching tags..."
git fetch --tags --quiet

if [ "$DRY_RUN" = 1 ]; then
  echo "[DRY RUN] Skipping branch switch (running in detached HEAD)..."
else
  echo "Switching to branch $CI_COMMIT_BRANCH..."
  git switch $CI_COMMIT_BRANCH
fi

echo "Logging in to GitLab..."
glab auth login --token $RELEASE_TOKEN

# Extract username from glab auth status
USERNAME=$(glab auth status 2>&1 | grep -E "Logged in to" | sed -E 's/.*✓ Logged in to [^ ]+ as ([^ ]+).*/\1/')

if [ -z "$USERNAME" ]; then
  echo "Could not extract username from glab auth status"
  exit 1
fi

echo "Using username: $USERNAME"

RELEASE_BOT_EMAIL="$USERNAME@noreply.gitlab.com"

echo "Creating SSH key..."
ssh-keygen -q -t rsa -b 4096 -C $RELEASE_BOT_EMAIL -f ~/.ssh/id_rsa -N ""
ssh-keyscan -q -t rsa gitlab.com >> ~/.ssh/known_hosts

echo "Adding SSH key to GitLab..."
glab ssh-key add ~/.ssh/id_rsa.pub -t "Login Key - $(date +%s)"

echo "Configuring git..."
git config --global gpg.format ssh
git config --global commit.gpgsign true
git config --global user.email "$RELEASE_BOT_EMAIL"
git config --global user.name "$RELEASE_BOT_NAME"
git config --global user.signingkey ~/.ssh/id_rsa
git config --global push.default current
git remote set-url origin $REMOTE_URL

if [ "$DRY_RUN" = 1 ]; then
  echo "[DRY RUN] Adding an empty feature commit..."
  git commit --allow-empty -m "feat: dry run empty feature" --quiet
fi

echo "Updating Cargo.toml and Cargo.lock..."
sed -i "s/^version = .*/version = \"$NEW_VERSION\"/" Cargo.toml
sed -i "/^name = \"glql\"/,/^\[/ s/^version = .*/version = \"$NEW_VERSION\"/" Cargo.lock

echo "Updating package.json..."
sed -i "s/\"version\": .*/\"version\": \"$NEW_VERSION\",/" package.json

# Note: Ruby gem version will be updated in a separate MR

echo "Installing Node.js, npm, and Cargo..."
apk add --update nodejs npm cargo --quiet

echo "Initial npm version: $(npm --version)"
echo "Upgrading npm to 11.6.2..."
npm install -g npm@11.6.2
echo "Updating PATH to use updated npm"
export PATH="/usr/local/bin:$PATH"

echo "Installed versions:"
echo "Node: $(node --version)"
echo "npm: $(npm --version)"
echo "Cargo: $(cargo --version)"

echo "Installing npm dependencies..."
npm i --silent

echo "Updating CHANGELOG.md..."
npm install --silent -g conventional-changelog conventional-changelog-conventionalcommits
conventional-changelog -i CHANGELOG.md -s -t v -k package.json -r 1 -n conventional-changelog.config.js -p conventionalcommits

echo "Committing changes..."
git add Cargo.toml Cargo.lock package.json package-lock.json CHANGELOG.md
git commit -m "chore(release): v$NEW_VERSION" --quiet

echo "Building npm package..."
npm run build

echo "Configuring npm..."
if [ -z "$NPM_ID_TOKEN" ]; then
  echo "NPM_ID_TOKEN is not defined. Cannot publish to npm."
  exit 1
fi
echo "@gitlab:registry=https://registry.npmjs.org/" >> ~/.npmrc

if [ "$DRY_RUN" = 1 ]; then
  echo "[DRY RUN] Publishing to npm..."
  npm publish --access public --dry-run
else
  echo "Publishing to npm..."
  npm publish --access public
fi

if [ "$DRY_RUN" = 1 ]; then
  echo "[DRY RUN] Skipping pushing changes..."
  echo "[DRY RUN] Would have pushed commit:"
  git show --stat
  echo ""
  echo "[DRY RUN] Diff:"
  git show
else
  echo "Pushing changes..."
  git push --quiet
fi

echo "Tagging release..."
git tag v$NEW_VERSION

if [ "$DRY_RUN" = 1 ]; then
  echo "[DRY RUN] Skipping pushing tags..."
else
  echo "Pushing tags..."
  git push --tags origin v$NEW_VERSION --quiet
fi

# Create MR for Ruby gem version bump. The MR requires the git release tag to be pushed first.

echo "Creating branch for Ruby gem version bump..."
RUBY_BRANCH="release/ruby-gem-v$NEW_VERSION"

if [ "$DRY_RUN" = 1 ]; then
  echo "[DRY RUN] Would create branch $RUBY_BRANCH"
else
  git checkout main
  git checkout -b "$RUBY_BRANCH"
fi

echo "Updating Ruby gem version.rb..."
sed -i "s/VERSION = .*/VERSION = \"$NEW_VERSION\"/" glql_rb/lib/gitlab_query_language/version.rb

echo "Updating Ruby Gemfile.lock..."
sed -i "/gitlab_query_language/s/([0-9]\+\.[0-9]\+\.[0-9]\+)/($NEW_VERSION)/" glql_rb/Gemfile.lock

echo "Updating Ruby gem GLQL dependency tag..."
sed -i "s/tag = \"v[0-9]\+\.[0-9]\+\.[0-9]\+\"/tag = \"v$NEW_VERSION\"/" glql_rb/ext/gitlab_query_language/Cargo.toml

if [ "$DRY_RUN" = 1 ]; then
  echo "[DRY RUN] Would Ruby gem update Cargo.lock (requires git tag to be pushed - which is skipped in dry run)"
else
  echo "Updating Ruby gem Cargo.lock..."
  cd glql_rb
  cargo update -p glql --quiet
  cd ..
fi

echo "Committing Ruby gem version changes..."
git add glql_rb/lib/gitlab_query_language/version.rb glql_rb/Gemfile.lock glql_rb/Cargo.lock glql_rb/ext/gitlab_query_language/Cargo.toml
git commit -m "chore(ruby): bump gem version to v$NEW_VERSION" --quiet

if [ "$DRY_RUN" = 1 ]; then
  echo "[DRY RUN] Skipping pushing Ruby branch..."
  echo "[DRY RUN] Would have pushed branch:"
  git show --stat
  echo ""
  echo "[DRY RUN] Diff:"
  git show
else
  echo "Pushing Ruby gem version branch..."
  git push -u origin "$RUBY_BRANCH" --quiet
fi

MR_TITLE="chore: bump gem version to v$NEW_VERSION"
MR_DESCRIPTION="This MR bumps the Ruby gem version to v$NEW_VERSION to trigger the gem release pipeline.

When merged to main, this will automatically trigger the gem-release jobs.

**Testing:** Add \`RELEASE\` (uppercase) to the MR title to enable the gem bundling manual jobs. You can trigger them manually to verify that the gem builds fine with the new version.

## Review Checklist

Please verify that version **v$NEW_VERSION** appears in all of the following files:
- [ ] \`glql_rb/lib/gitlab_query_language/version.rb\`
- [ ] \`glql_rb/Gemfile.lock\`
- [ ] \`glql_rb/Cargo.lock\`
- [ ] \`glql_rb/ext/gitlab_query_language/Cargo.toml\` (tag reference)

Related npm release: v$NEW_VERSION"

REVIEWER_FLAGS=""
for reviewer in $REVIEWERS; do
  REVIEWER_FLAGS="$REVIEWER_FLAGS --reviewer $reviewer"
done

if [ "$DRY_RUN" = 1 ]; then
  echo "[DRY RUN] Would create MR for Ruby gem version bump"
  echo ""
  echo "Title: $MR_TITLE"
  echo "Assignee: "$USERNAME""
  echo "Reviewers flags: $REVIEWER_FLAGS"
  echo "Description:"
  echo "$MR_DESCRIPTION"
else
  echo "Creating MR for Ruby gem version bump..."

  glab mr create \
    --title "$MR_TITLE" \
    --description "$MR_DESCRIPTION" \
    --assignee "$USERNAME" \
    $REVIEWER_FLAGS \
    --source-branch "$RUBY_BRANCH" \
    --target-branch main \
    --remove-source-branch \
    --yes

  echo "✓ Ruby gem version MR created successfully!"
  echo "  Please review and merge the MR to trigger the gem release."
fi