import { formatSdl } from 'format-graphql';
import { resetUniqueIdCounters } from '../src/utils/common';
import { compileToGraphql, mockCurrentDate, transformData } from './test_utils';

describe('graphql:', () => {
  beforeEach(mockCurrentDate);
  beforeEach(resetUniqueIdCounters);

  it.each([
    'weight = 1',
    'assignee = currentUser()',
    'label in ("devops::plan", "devops::create")',
    'label != "backend" and author = currentUser() and weight = 1 and updated > today()',
    'weight = 1 and updated > startOfDay("-7")',
    'updated > today()',
    'health = "on track"',
    'health = "needs attention"',
    'health = "at risk"',
    'iteration = 123',
    'cadence = 123 and iteration = current',
    'group = "gitlab-test" and weight = 1 and includeSubgroups = false',
  ])("compiles '%s' with default fields: ['id', 'title']", async (query: string) => {
    const compiled = await compileToGraphql(query);

    expect(compiled.output).toMatchSnapshot();
  });

  it.each([
    'epic = 123',
    'epic = gitlab-test&123',
    'epic = &123',
    'epic = "123"',
    'type = Epic and author = @foo and updated > today()',
  ])("compiles epic query '%s' with default fields: ['id', 'title']", async (query: string) => {
    const compiled = await compileToGraphql(query);

    for (const key in compiled.variables) {
      compiled.variables[key].value = compiled.variables[key].value
        ? formatSdl(compiled.variables[key].value)
        : null;
    }

    expect({
      output: compiled.output,
      variables: compiled.variables,
    }).toMatchSnapshot();
  });

  it('support IN with epic in the work items API', async () => {
    const compiled = await compileToGraphql(
      'epic in (gitlab-test&1, gitlab-test&2, gitlab-com&3, &4)',
      { useWorkItemsAPI: true },
    );

    for (const key in compiled.variables) {
      compiled.variables[key].value = compiled.variables[key].value
        ? formatSdl(compiled.variables[key].value)
        : null;
    }

    expect({
      output: compiled.output,
      variables: compiled.variables,
    }).toMatchSnapshot();
  });

  it.each`
    field
    ${'labels'}
    ${'assignees'}
    ${'milestone'}
    ${'iteration'}
    ${'lastComment'}
    ${'type'}
  `('compiles query with field: $field', async ({ field }: { field: string }) => {
    const compiled = await compileToGraphql('weight = 1', {
      fields: ['id', field],
    });

    expect(compiled.output).toMatchSnapshot();
  });

  it.each`
    field
    ${'iid'}
    ${'title'}
    ${'webUrl'}
    ${'assignees'}
    ${'labels'}
    ${'milestone'}
    ${'weight'}
    ${'healthStatus'}
    ${'startDate'}
    ${'dueDate'}
    ${'iteration'}
    ${'progress'}
    ${'epic'}
    ${'parent'}
    ${'color'}
    ${'taskCompletionStatus'}
    ${'lastComment'}
    ${'timeEstimate'}
    ${'totalTimeSpent'}
    ${'workItemType'}
    ${'type'}
  `(
    'compiles query with field: $field with the work items API',
    async ({ field }: { field: string }) => {
      const compiled = await compileToGraphql('weight = 1', {
        fields: ['id', field],
        useWorkItemsAPI: true,
      });

      expect(compiled.output).toMatchSnapshot();
    },
  );

  it('compiles query with sort', async () => {
    const compiled = await compileToGraphql('weight = 1', {
      sort: 'dueDate asc',
    });

    expect(compiled.output).toMatchSnapshot();
  });

  it('does not use work items API for merge requests', async () => {
    const compiled = await compileToGraphql('type = MergeRequest', {
      fields: ['id', 'labels'],
      useWorkItemsAPI: true,
    });

    expect(compiled.output).toMatchSnapshot();
  });

  it('compiles query with fields: startDate and dueDate with the work items API', async () => {
    const compiled = await compileToGraphql('weight = 1', {
      fields: ['id', 'startDate', 'dueDate'],
      useWorkItemsAPI: true,
    });

    expect(compiled.output).toMatchSnapshot();
  });

  it('compiles query with fields: timeEstimate and totalTimeSpent with the work items API', async () => {
    const compiled = await compileToGraphql('weight = 1', {
      fields: ['id', 'timeEstimate', 'totalTimeSpent'],
      useWorkItemsAPI: true,
    });

    expect(compiled.output).toMatchSnapshot();
  });

  it.each`
    query
    ${'group = "gitlab-test" and weight = 1 and includeSubgroups = false'}
    ${'health = "on track"'}
  `('compiles query `$query` correctly for work items', async ({ query }: { query: string }) => {
    const compiled = await compileToGraphql(query, {
      fields: ['id', 'title'],
      useWorkItemsAPI: true,
    });

    expect(compiled.output).toMatchSnapshot();
  });
});

describe('transform', () => {
  it('transforms data', async () => {
    const mockData = {
      project: {
        issues: {
          nodes: [
            {
              id: 1,
              title: 'test',
              labels: {
                nodes: [
                  { title: 'workflow::in progress' },
                  { title: 'type::bug' },
                  { title: 'frontend' },
                  { title: 'backend' },
                ],
              },
              lastComment: { nodes: [{ bodyHtml: 'test' }] },
            },
          ],
          count: 400,
        },
      },
    };
    const { data, fields } = await transformData(mockData, {
      fields: [
        'id',
        'title',
        'labels("workflow::*") as "Workflow"',
        "labels('type::*') as 'Type'",
        "labels as 'Remaining labels'",
        "lastComment as 'THE LAST COMMENT'",
      ],
    });

    expect(data).toMatchSnapshot();
    expect(fields).toMatchSnapshot();
  });
});
