import { compile } from '../../src/lib';
import { featureFlags } from '../../src/utils/feature_flags';
import { compileWithProjectContext } from './test_helpers';

test('type equals merge request', () => {
  // raw parse test
  expect(
    compile('type = MergeRequest', {
      fields: 'id, title',
      project: 'gitlab-org/gitlab',
    }).output.split('\n'),
  ).toContain('    mergeRequests(before: $before, after: $after, first: $limit) {');

  expect(
    compileWithProjectContext('gitlab-org/gitlab', 'type = MergeRequest and state = merged').split(
      '\n',
    ),
  ).toContain('    mergeRequests(state: merged, before: $before, after: $after, first: $limit) {');

  expect(
    compileWithProjectContext(
      'gitlab-org/gitlab',
      'reviewer = @foo and type = MergeRequest and state = merged',
    ).split('\n'),
  ).toEqual(
    expect.arrayContaining([
      '    mergeRequests(',
      '      reviewerUsername: "foo"',
      '      state: merged',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('type equals issue', () => {
  expect(
    compileWithProjectContext('gitlab-org/gitlab', 'type = Issue and state = opened').split('\n'),
  ).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      types: ISSUE',
      '      state: opened',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  expect(
    compileWithProjectContext('gitlab-org/gitlab', 'confidential = true and type = Issue').split(
      '\n',
    ),
  ).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      confidential: true',
      '      types: ISSUE',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('type equals work item type', () => {
  expect(
    compileWithProjectContext('gitlab-org/gitlab', 'type = Objective and state = opened').split(
      '\n',
    ),
  ).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      types: OBJECTIVE',
      '      state: opened',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  expect(
    compileWithProjectContext('gitlab-org/gitlab', 'type = KeyResult and state = opened').split(
      '\n',
    ),
  ).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      types: KEY_RESULT',
      '      state: opened',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('type equals invalid', () => {
  expect(compileWithProjectContext('gitlab-org/gitlab', 'state = opened and type = nOnE')).toEqual(
    'Error: `type` cannot be compared with `nOnE`. Supported value types: `Enum` (`Issue`, `Incident`, `TestCase`, `Requirement`, `Task`, `Ticket`, `Objective`, `KeyResult`, `Epic`, `MergeRequest`), `List`.',
  );

  expect(compileWithProjectContext('gitlab-org/gitlab', 'label = "fish" and type = Any')).toEqual(
    'Error: `type` cannot be compared with `Any`. Supported value types: `Enum` (`Issue`, `Incident`, `TestCase`, `Requirement`, `Task`, `Ticket`, `Objective`, `KeyResult`, `Epic`, `MergeRequest`), `List`.',
  );
});

test('type equals valid but invalid fields', () => {
  expect(
    compileWithProjectContext('gitlab-org/gitlab', 'reviewer = @foo and type = Issue'),
  ).toEqual('Error: `reviewer` is not a recognized field for work items.');

  expect(
    compileWithProjectContext('gitlab-org/gitlab', 'confidential = true and type = MergeRequest'),
  ).toEqual('Error: `confidential` is not a recognized field for merge requests.');
});

test('type in work item types', () => {
  expect(
    compileWithProjectContext('gitlab-org/gitlab', 'type in (Issue, Incident, TestCase)').split(
      '\n',
    ),
  ).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      types: [ISSUE, INCIDENT, TEST_CASE]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );

  expect(
    compileWithProjectContext('gitlab-org/gitlab', 'type in (Objective, KeyResult)').split('\n'),
  ).toEqual(
    expect.arrayContaining([
      '    issues(',
      '      types: [OBJECTIVE, KEY_RESULT]',
      '      before: $before',
      '      after: $after',
      '      first: $limit',
      '    ) {',
    ]),
  );
});

test('type in work item types with epic', () => {
  // with FF enabled: allowed
  featureFlags.glqlWorkItems = true;

  expect(
    compileWithProjectContext('gitlab-org/gitlab', 'type in (Epic, Issue)').split('\n'),
  ).toEqual(
    expect.arrayContaining([
      '    workItems(types: [EPIC, ISSUE], before: $before, after: $after, first: $limit) {',
    ]),
  );

  // with FF disabled: not allowed
  featureFlags.glqlWorkItems = false;
  expect(compileWithProjectContext('gitlab-org/gitlab', 'type in (Epic, Issue)')).toEqual(
    'Error: Type `Epic` cannot be combined with other types (`Issue`) in the same query. Try using `type = Epic` instead.',
  );
});
